/* SocketOpts.java

   COPYRIGHT 2010 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.xmp;

import java.util.Date;
import java.util.Calendar;
import java.net.*;
import javax.net.*;
import javax.net.ssl.*;
import javax.net.ssl.SSLServerSocketFactory;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.Signature;
import java.security.cert.X509Certificate;
import java.security.SecureRandom;
import java.math.BigInteger;
import java.io.FileOutputStream;

// internal sun APIs; warnings galore
import java.security.cert.*;
import sun.security.x509.*;

// for Java 8, CertKeyGen is located in a different package
// 
//import sun.security.tools.keytool.CertAndKeyGen;

/** 
 *  Encapsulate Socket, SSL, communication options, parameters, etc.
 *  @version $Id: SocketOpts.java 15 2008-07-17 14:20:37Z rdk $
 *  @see XmpSession
 *  @see XmpMessage
 *  @see Xmp
 **/

public class SocketOpts {

   /* class variables and methods *********************** */
   public static final int defaultConnectTimeout = 3000; /* ms */
   public static int defaultMaxBacklog = 50;
   public static boolean defaultReuseAddr = true;
   public static final int defaultReadTimeout = 15*1000; /* ms */

   private static final int keysize=1024;
   private static final String commonName = "FQDN";
   private static final String orgUnit = "Cartographer";
   private static final String orgName = "Krupczak.org, LLC";
   private static final String stateName = "Georgia";
   private static final String localityName = "Smyrna";
   private static final String countryName = "US";
   private static final char[] password = {'c','a','r','t','o','g','r','a','p','h','e','r'}; 
   private static final char[] passwd1 = {'c','h','a','n','g','e','i','t'};

   /* instance variables ******************************** */
   public SSLContext sslContext;
   public SocketFactory sslSocketFactory;
   public ServerSocketFactory sslServerSocketFactory;

   public SSLSessionContext serverSessionContext;

   public boolean keystoreFound;
   public int connectTimeout;
   public int maxBacklog;
   public boolean reuseAddr;
   public int readTimeout;
   public int ipVersion;

   /* constructors  ************************************* */
   public SocketOpts() 
   { 

     // initialize SSL parameters and context
     // the trust store should be per-context rather than
     // the entire java vm which could break other things
     // if we provide a JAR file for use within a larger
     // software package

     ClassLoader cl;
     java.io.InputStream kis = null;
     KeyStore ks;
     //java.io.InputStream kis1 = null;
     //KeyStore ks1;

     TrustManager tm[];
     KeyManager km[] = null;

     cl = getClass().getClassLoader();

     try {

       // load up our Cartographer keystore for evaluating
       // our certs and create a Trust manager
       if (isAndroid()) {
          ks = KeyStore.getInstance("BKS");
       }
       else {
          ks = KeyStore.getInstance("jks");
       }
       //ks1 = KeyStore.getInstance("jks");
       
       if (cl != null) {
          if (isAndroid()) {
             kis = cl.getResourceAsStream("org/krupczak/xmp/Cartographer.bks");
             if (kis == null) {
                kis = ClassLoader.getSystemResourceAsStream("org/krupczak/xmp/Cartographer.bks");
	     }
	  } else {
             kis = cl.getResourceAsStream("org/krupczak/xmp/Cartographer.keystore");
             if (kis == null) {
                kis = ClassLoader.getSystemResourceAsStream("org/krupczak/xmp/Cartographer.keystore");
	     }
	  }
       }
       else {
	   if (isAndroid()) {
              kis = ClassLoader.getSystemResourceAsStream("org/krupczak/xmp/Cartographer.bks");
	   }
	   else {
              kis = ClassLoader.getSystemResourceAsStream("org/krupczak/xmp/Cartographer.keystore");
	   }
       }

       // if we dont pull it out of our JAR, try it in some well-known
       // locations
       //if (kis == null) {
       //   kis = new java.io.FileInputStream("etc/Cartographer.keystore");
       //}

       ks.load(kis,password);
       if (kis != null) {
          kis.close();
       }
       else {
	   System.out.println("SocketOpts: unable to find keystore");
       }

       //ks1.load(kis1,password);
       //if (kis1 != null) {
       //   kis1.close();
       //}
       //else {
       //   System.out.println("SocketOpts: unable to find key keystore");
       //}

       //java.io.FileInputStream fis = new java.io.FileInputStream("Cartographer.keystore");
       //ks.load(fis,password);
       //fis.close();

       TrustManagerFactory tmf;
       if (isAndroid()) {
          tmf = TrustManagerFactory.getInstance("X509");
       }
       else {
          tmf = TrustManagerFactory.getInstance("SunX509");
       }
       tmf.init(ks);
       tm = tmf.getTrustManagers();
       keystoreFound = true;

       // generate our own credentials keys/cert
       km = null;
       //km = generateSignedCredentials(gethostname(),ks,ks1);

       if (isAndroid() == false) 
          km = generateSignedCredentials(gethostname(),ks,ks);

     } catch (Exception e) {
        ks = null;
        tm = null;
        System.out.println("SocketOpts: Could not allocate trust manager keystore");
        System.out.println("SocketOpts: Exception "+e.getMessage());
        e.printStackTrace();

        // this sets our Certificate authority
	if (isAndroid()) {
           System.setProperty("javax.net.ssl.trustStore","Cartographer.bks");
	}
	else {
           System.setProperty("javax.net.ssl.trustStore","Cartographer.keystore");
	}
        System.setProperty("javax.net.ssl.trustStorePassword","cartographer");
        System.setProperty("javax.net.debug","ssl");
        keystoreFound = false;
     }

     try {
         // initialize our SSL context with our own cert/key-pair/manager
         // and the trust/CA manager
         // SSLv3 < TLSv1
	 sslContext = SSLContext.getInstance("SSLv3");
         //sslContext.init(null,tm,null);
         sslContext.init(km,tm,null);
         sslSocketFactory = sslContext.getSocketFactory();
         serverSessionContext = sslContext.getServerSessionContext();
         sslServerSocketFactory = sslContext.getServerSocketFactory();

         //System.out.println("SocketOpts: obtained SSL socket factories");

     } catch (Exception e) {
         System.out.println("SocketOpts: using default ssl socket factories");
         sslSocketFactory = SSLSocketFactory.getDefault(); 
         sslServerSocketFactory = SSLServerSocketFactory.getDefault();
     }

     connectTimeout = defaultConnectTimeout;
     maxBacklog = defaultMaxBacklog;
     reuseAddr = defaultReuseAddr;
     readTimeout = defaultReadTimeout;

     //System.out.println("SockOpts constructor finished");

     return; 
   }

   /* private methods *********************************** */
   private String gethostname() 
   {
       String myHostName, myIpAddr;

       try {
         myHostName = InetAddress.getLocalHost().getCanonicalHostName();
         myIpAddr = InetAddress.getLocalHost().getHostAddress();

       } catch (UnknownHostException e) {
            myHostName = "127.0.0.1"; 
            myIpAddr = "127.0.0.1";
       }

       return myHostName;

   } /* gethostname() */

   // generate a key-pair and signed certificate for use with
   // Xmp sessions
   private KeyManager[] generateSelfSignedCredentials(String hostname)
   {
       KeyStore keyStore;
       //CertAndKeyGen keypair;
       KeyPair keypair;
       KeyPairGenerator kpgen;
       Date notBefore, notAfter;
       Calendar cal = Calendar.getInstance();
       Signature rsa;

       try {
         
         // create empty keystore
         keyStore = KeyStore.getInstance("JKS");
         keyStore.load(null,null);

         // generate a key pair
         //keypair = new CertAndKeyGen("RSA","SHA1WithRSA",null);
         //keypair.generate(keysize);
         kpgen = KeyPairGenerator.getInstance("RSA");
         kpgen.initialize(keysize);
         keypair = kpgen.generateKeyPair();

         rsa = Signature.getInstance("SHA1withRSA");
         rsa.initSign(keypair.getPrivate());

         // generate x500 name
         // we'd like to add extension FQND,hostname XXX
         X500Name x500Name = new X500Name(hostname,
                                        orgUnit,
                                        orgName,
                                        localityName,
                                        stateName,
                                        countryName);
       
         // create the self cert right now
         // we'd like to sign it with Cartographer CA XXX
         notBefore = new Date();
         cal.add(Calendar.YEAR,5);
         notAfter = cal.getTime();
         X509Certificate[] chain = new X509Certificate[1];

         // can't call getSelfCertificate() on keypair
         // only on CertAndKeyGen which is deprecated in Java 8
         // there is code to generate self-signed cert
         // out on net but we don't need it
         // chain[0] = keypair.getSelfCertificate(x500Name,
         //                                    notBefore,
         //                                    5*365*24*60*60 //secs?
         //                                    );

         // stick keys and cert in the keystore
         keyStore.setKeyEntry("xmpentity",
                            (PrivateKey)keypair.getPrivate(),
                            password,
                            chain);

         // make keymanager out of keystore
         KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
         kmf.init(keyStore,password);

         KeyManager km[];
         km = kmf.getKeyManagers();

         return km;

       } catch (Exception e) { 
           System.out.println("generateSelfSignedCredentials: exception"+e); 
           e.printStackTrace();
           return null; 
       }

   } /* create creditionals */

   private void writeNewKeystore(Certificate caCert, PrivateKey caPrivKey)
   {
       KeyStore ks;
       Certificate[] chain = new Certificate[1];
       chain[0] = caCert;

       try {
         ks = KeyStore.getInstance("JKS");
         ks.load(null,null);

         ks.setKeyEntry("ca",caPrivKey,password,chain);

         FileOutputStream fos = null;
         fos = new FileOutputStream("newKeyStore.keystore");
         ks.store(fos,password);
         if (fos != null) fos.close();

       } catch (Exception e) {
           System.out.println("writeNewKeystore: exception "+e);
       }

   }

   private KeyManager[] generateSignedCredentials0(String fqdn, 
                                                  KeyStore caKeyStore,
                                                  KeyStore caPrivKeyStore)
   {
       KeyStore keyStore;
       //CertAndKeyGen keypair;
       KeyPair keypair;
       KeyPairGenerator kpgen;
       Date notBefore, notAfter;
       Calendar cal = Calendar.getInstance();
       String algorithm = new String("SHA1withRSA");

       try {

         // get info from our CA keystore
         Certificate caCert = caKeyStore.getCertificate("ca");
         byte[] encoded = caCert.getEncoded();
         X509CertImpl caCertImpl = new X509CertImpl(encoded);
         X509CertInfo caCertInfo = (X509CertInfo)caCertImpl.get(X509CertImpl.NAME+"."+X509CertImpl.INFO);
         X500Name issuer = (X500Name)caCertInfo.get(X509CertInfo.SUBJECT+"."+CertificateIssuerName.DN_NAME);

         // get private key from our key store
	 PrivateKey caPrivateKey = (PrivateKey)caPrivKeyStore.getKey("ca",
                                                                     password);

         String dn = "C=US, ST=Georgia, L=Smyrna, O=Krupczak.org, OU=Cartographer, CN="+fqdn;
         
         // create empty keystore
         keyStore = KeyStore.getInstance("JKS");
         keyStore.load(null,null);

         // generate a key pair
         //keypair = new CertAndKeyGen("RSA","SHA1WithRSA",null);
         //keypair.generate(keysize);
         kpgen = KeyPairGenerator.getInstance("RSA");
         kpgen.initialize(keysize);
         keypair = kpgen.generateKeyPair();


         // generate the certificate
         X509CertInfo info = new X509CertInfo();
         notBefore = new Date();
         cal.add(Calendar.YEAR,5);
         notAfter = new Date(notBefore.getTime() + 5*365 * 86400000l);
         CertificateValidity interval = new CertificateValidity(notBefore,notAfter);
         BigInteger sn = new BigInteger(64, new SecureRandom());
         X500Name owner = new X500Name(dn); // my Xmp protocol entity
         info.set(X509CertInfo.VALIDITY,interval);
         info.set(X509CertInfo.SERIAL_NUMBER,new CertificateSerialNumber(sn));
         info.set(X509CertInfo.SUBJECT,new CertificateSubjectName(owner));

         // issuer is our Cartographer CA
         info.set(X509CertInfo.ISSUER,new CertificateIssuerName(issuer));

         info.set(X509CertInfo.KEY,new CertificateX509Key(keypair.getPublic()));
         info.set(X509CertInfo.VERSION,new CertificateVersion(CertificateVersion.V3));
         AlgorithmId algo = new AlgorithmId(AlgorithmId.md5WithRSAEncryption_oid);
         info.set(X509CertInfo.ALGORITHM_ID, new CertificateAlgorithmId(algo));

         // set Subject Alt Name extension and others if applicable
         CertificateExtensions ext = new CertificateExtensions();
         DNSName dnsName = new DNSName(fqdn);
         GeneralName generalName = new GeneralName(dnsName);
         GeneralNames generalNames = new GeneralNames();
         generalNames.add(generalName);
         ext.set(SubjectAlternativeNameExtension.NAME,
		 new SubjectAlternativeNameExtension(generalNames));
         ext.set(BasicConstraintsExtension.NAME, 
                 new BasicConstraintsExtension(false,1));
         info.set(X509CertInfo.EXTENSIONS,ext);

         // sign the cert to identify the algorithm thats used
         X509CertImpl cert = new X509CertImpl(info);
         cert.sign(caPrivateKey, algorithm);
         //cert.sign(bogusKeypair.getPrivate(), algorithm);
 
         // Update the algorith, and resign.
         algo = (AlgorithmId)cert.get(X509CertImpl.SIG_ALG);
         info.set(CertificateAlgorithmId.NAME + "." + 
                  CertificateAlgorithmId.ALGORITHM,algo);
         cert = new X509CertImpl(info);
         //cert.sign(bogusKeypair.getPrivate(), algorithm);
         cert.sign(caPrivateKey, algorithm);

         X509Certificate[] chain = new X509Certificate[1];
         chain[0] = cert;

         // store them in keystore
         keyStore.setKeyEntry("XmpEntity",
                            (PrivateKey)keypair.getPrivate(),
                            password,
                            chain);

         // write new keystore?
         //writeNewKeystore(caCert,caPrivateKey);

         // make keymanager out of keystore
         KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
         kmf.init(keyStore,password);

         KeyManager km[];
         km = kmf.getKeyManagers();

         return km;

       } catch (Exception e) { 
           System.out.println("generateSignedCredentials: exception "+e); 
           e.printStackTrace();
           return null; 
       }
   }

   private KeyManager[] generateSignedCredentials(String fqdn, 
                                                  KeyStore caKeyStore,
                                                  KeyStore caPrivKeyStore)
   {
       KeyStore keyStore;
       //CertAndKeyGen keypair;
       KeyPair keypair;
       KeyPairGenerator kpgen;
       Date notBefore, notAfter;
       Calendar cal = Calendar.getInstance();
       String algorithm = new String("SHA1withRSA");

       try {

         // get info from our CA keystore
         Certificate caCert = caKeyStore.getCertificate("ca");
         byte[] encoded = caCert.getEncoded();
         X509CertImpl caCertImpl = new X509CertImpl(encoded);
         X509CertInfo caCertInfo = (X509CertInfo)caCertImpl.get(X509CertImpl.NAME+"."+X509CertImpl.INFO);
         X500Name issuer = (X500Name)caCertInfo.get(X509CertInfo.SUBJECT+"."+CertificateIssuerName.DN_NAME);

         // get private key from our key store
	 PrivateKey caPrivateKey = (PrivateKey)caPrivKeyStore.getKey("ca",
                                                                     password);

         // distinguished name
         String dn = "C=US, ST=Georgia, L=Smyrna, O=Krupczak.org, OU=Cartographer, CN="+fqdn;
         
         // create empty keystore
         keyStore = KeyStore.getInstance("JKS");
         keyStore.load(null,null);

         // generate a key pair
         // keypair = new CertAndKeyGen("RSA","SHA1WithRSA",null);
         // keypair.generate(keysize);
         kpgen = KeyPairGenerator.getInstance("RSA");
         kpgen.initialize(keysize);
         keypair = kpgen.generateKeyPair();

         // generate the certificate request
         X509CertInfo info = new X509CertInfo();

         // make notBefore=yesterday to deal with vagaries of 
         // clock synchronization and GMT offsets and
         // computers out of sync
         notBefore = new Date(cal.getTimeInMillis() - 86400000l);
         cal.add(Calendar.YEAR,5);
         notAfter = new Date(notBefore.getTime() + 5*365 * 86400000l);
         CertificateValidity interval = 
                             new CertificateValidity(notBefore,notAfter);
         BigInteger sn = new BigInteger(64, new SecureRandom());
         X500Name owner = new X500Name(dn); // my Xmp protocol entity
         CertificateSubjectName csn = new CertificateSubjectName(owner);

         //System.out.println("CertificateSubjectName "+csn);
         //System.out.println("CertificateSubjectName class "+csn.getClass());
         //System.out.println("X500Name "+owner);
         //System.out.println("X500Name is class "+owner.getClass());

         info.set(X509CertInfo.VALIDITY,interval);
         info.set(X509CertInfo.SERIAL_NUMBER,new CertificateSerialNumber(sn));

         try {
           info.set(X509CertInfo.SUBJECT,csn);
         } catch (Exception xE) {
	     //System.out.println("Exception setting Certificate Subject");
             //System.out.println(xE);
             //System.out.println("Trying again with X500Name");
             info.set(X509CertInfo.SUBJECT,owner);
             //System.out.println("Retry with X500Name seemed to work");
         }

         // issuer is our Cartographer CA
         try {
	   info.set(X509CertInfo.ISSUER,new CertificateIssuerName(issuer));
         } catch (Exception xE) {
	     //System.out.println("Exception setting Certificate Issuer");
             //System.out.println(xE);
             //System.out.println("Trying again with X500Name");
  	     info.set(X509CertInfo.ISSUER,issuer);
             //System.out.println("Retry with X500Name seemed to work");
         }

         try {
           info.set(X509CertInfo.KEY,
                    new CertificateX509Key(keypair.getPublic()));
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate Key");
             System.out.println(xE);
         }

         try {
           info.set(X509CertInfo.VERSION,
                    new CertificateVersion(CertificateVersion.V3));
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate version");
             System.out.println(xE);
         }

         AlgorithmId algo = new AlgorithmId(AlgorithmId.md5WithRSAEncryption_oid);
         info.set(X509CertInfo.ALGORITHM_ID, new CertificateAlgorithmId(algo));

         // set Subject Alt Name extension and others if applicable
         CertificateExtensions ext = new CertificateExtensions();
         DNSName dnsName = new DNSName(fqdn);
         GeneralName generalName = new GeneralName(dnsName);
         GeneralNames generalNames = new GeneralNames();
         generalNames.add(generalName);
         ext.set(SubjectAlternativeNameExtension.NAME,
		 new SubjectAlternativeNameExtension(generalNames));
         ext.set(BasicConstraintsExtension.NAME, 
                 new BasicConstraintsExtension(false,1));
         info.set(X509CertInfo.EXTENSIONS,ext);

         // generate and sign cert request to identify the algorithm thats used
         X509CertImpl cert = new X509CertImpl(info);
         cert.sign(caPrivateKey, algorithm);
         //cert.sign(keypair.getPrivate, algorithm);
 
         // Update the algorithm, and resign with CA cert private key
         // for the real certificate
         algo = (AlgorithmId)cert.get(X509CertImpl.SIG_ALG);
         info.set(CertificateAlgorithmId.NAME + "." + 
                  CertificateAlgorithmId.ALGORITHM,algo);
         cert = new X509CertImpl(info);
         cert.sign(caPrivateKey, algorithm);

         X509Certificate[] chain = new X509Certificate[1];
         chain[0] = cert;

         // store them in keystore
         keyStore.setKeyEntry("XmpEntity",
                            (PrivateKey)keypair.getPrivate(),
                            password,
                            chain);

         // write new keystore?
         //writeNewKeystore(caCert,caPrivateKey);

         // make keymanager out of keystore
         KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
         kmf.init(keyStore,password);

         KeyManager km[];
         km = kmf.getKeyManagers();

         return km;

       } catch (Exception e) { 
           System.out.println("generateSignedCredentials: exception "+e); 
           e.printStackTrace();
           return null; 
       }
   }

   private KeyManager[] generateSignedCredentialsNew(String fqdn, 
                                                  KeyStore caKeyStore,
                                                  KeyStore caPrivKeyStore)
   {
       KeyStore keyStore;
       //CertAndKeyGen keypair;
       KeyPair keypair;
       KeyPairGenerator kpgen;
       Date notBefore, notAfter;
       Calendar cal = Calendar.getInstance();
       String algorithm = new String("SHA1withRSA");

       try {

         // get info from our CA keystore
         Certificate caCert = caKeyStore.getCertificate("ca");
         byte[] encoded = caCert.getEncoded();
         X509CertImpl caCertImpl = new X509CertImpl(encoded);
         X509CertInfo caCertInfo = (X509CertInfo)caCertImpl.get(X509CertImpl.NAME+"."+X509CertImpl.INFO);
         X500Name issuer = (X500Name)caCertInfo.get(X509CertInfo.SUBJECT+"."+CertificateIssuerName.DN_NAME);

         // get private key from our key store
	 PrivateKey caPrivateKey = (PrivateKey)caPrivKeyStore.getKey("ca",
                                                                     password);

         // distinguished name
         String dn = "C=US, ST=Georgia, L=Smyrna, O=Krupczak.org, OU=Cartographer, CN="+fqdn;
         
         // create empty keystore
         keyStore = KeyStore.getInstance("JKS");
         keyStore.load(null,null);

         // generate a key pair
         // keypair = new CertAndKeyGen("RSA","SHA1WithRSA",null);
         // keypair.generate(keysize);
         kpgen = KeyPairGenerator.getInstance("RSA");
         kpgen.initialize(keysize);
         keypair = kpgen.generateKeyPair();

         // generate the certificate request
         X509CertInfo info = new X509CertInfo();

         // make notBefore=yesterday to deal with vagaries of 
         // clock synchronization and GMT offsets and
         // computers out of sync
         notBefore = new Date(cal.getTimeInMillis() - 86400000l);
         cal.add(Calendar.YEAR,5);
         notAfter = new Date(notBefore.getTime() + 5*365 * 86400000l);
         CertificateValidity interval = 
                             new CertificateValidity(notBefore,notAfter);
         BigInteger sn = new BigInteger(64, new SecureRandom());
         X500Name owner = new X500Name(dn); // my Xmp protocol entity
         CertificateSubjectName csn = new CertificateSubjectName(owner);

         System.out.println("CertificateSubjectName "+csn);
         System.out.println("CertificateSubjectName is class "+csn.getClass());
         info.set(X509CertInfo.VALIDITY,interval);
         info.set(X509CertInfo.SERIAL_NUMBER,new CertificateSerialNumber(sn));

         try {
           info.set(X509CertInfo.SUBJECT,csn);
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate Subject");
             System.out.println(xE);
         }

         // issuer is our Cartographer CA
         try {
	   info.set(X509CertInfo.ISSUER,new CertificateIssuerName(issuer));
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate Issuer");
             System.out.println(xE);
         }

         try {
           info.set(X509CertInfo.KEY,
                    new CertificateX509Key(keypair.getPublic()));
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate Key");
             System.out.println(xE);
         }

         try {
           info.set(X509CertInfo.VERSION,
                    new CertificateVersion(CertificateVersion.V3));
         } catch (Exception xE) {
	     System.out.println("Exception setting Certificate version");
             System.out.println(xE);
         }

         AlgorithmId algo = new AlgorithmId(AlgorithmId.md5WithRSAEncryption_oid);
         info.set(X509CertInfo.ALGORITHM_ID, new CertificateAlgorithmId(algo));

         // set Subject Alt Name extension and others if applicable
         CertificateExtensions ext = new CertificateExtensions();
         DNSName dnsName = new DNSName(fqdn);
         GeneralName generalName = new GeneralName(dnsName);
         GeneralNames generalNames = new GeneralNames();
         generalNames.add(generalName);
         ext.set(SubjectAlternativeNameExtension.NAME,
		 new SubjectAlternativeNameExtension(generalNames));
         ext.set(BasicConstraintsExtension.NAME, 
                 new BasicConstraintsExtension(false,1));
         info.set(X509CertInfo.EXTENSIONS,ext);

         // generate and sign cert request to identify the algorithm thats used
         X509CertImpl cert = new X509CertImpl(info);
         cert.sign(caPrivateKey, algorithm);
         //cert.sign(keypair.getPrivate, algorithm);
 
         // Update the algorithm, and resign with CA cert private key
         // for the real certificate
         algo = (AlgorithmId)cert.get(X509CertImpl.SIG_ALG);
         info.set(CertificateAlgorithmId.NAME + "." + 
                  CertificateAlgorithmId.ALGORITHM,algo);
         cert = new X509CertImpl(info);
         cert.sign(caPrivateKey, algorithm);

         X509Certificate[] chain = new X509Certificate[1];
         chain[0] = cert;

         // store them in keystore
         keyStore.setKeyEntry("XmpEntity",
                            (PrivateKey)keypair.getPrivate(),
                            password,
                            chain);

         // write new keystore?
         //writeNewKeystore(caCert,caPrivateKey);

         // make keymanager out of keystore
         KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
         kmf.init(keyStore,password);

         KeyManager km[];
         km = kmf.getKeyManagers();

         return km;

       } catch (Exception e) { 
           System.out.println("generateSignedCredentials: exception "+e); 
           e.printStackTrace();
           return null; 
       }
   }

   /* public methods ************************************ */
   public void setReadTimeout(int newTimeout)
   {
       if (newTimeout >= 0)
	   readTimeout = newTimeout;

      /* timeouts of 0 effectively means block for infinite
         amount of time
      */

   }

   public int getReadTimeout() { return readTimeout; }

   public void setConnectTimeout(int newTimeout) 
   {
       if (newTimeout >= 0)
	   connectTimeout = newTimeout;

      /* timeouts of 0 effectively means block for a long time
         the deault timeout, when calling socket() w/o one, is
         unknown 
      */
   }

   public int getConnectTimeout() { return connectTimeout; }

   // did we correctly find our keystore and load it up? 
   public boolean getKeystoreFound() { return keystoreFound; }

   // what is the max number of connection backlog we'd like
   // for server sessions only
   public int getMaxBacklog() { return maxBacklog; }

   // set maximum connection backlog we'd like 
   // for server sessions only
   public void setMaxBacklog(int b) { maxBacklog = b; }

   // should server socket reuseAddr?
   public boolean getReuseAddr() { return reuseAddr; }
   public void setReuseAddr(boolean b) { reuseAddr = b; }

   // set the private key we are using for server sessions

   // set the cert that we are using for server sessions
   // this cert is signed by our Cartographer CA

   // are we running on Android?
   public boolean isAndroid()
   {
       String runtime;

       runtime = System.getProperty("java.runtime.name");
       if (runtime.contains("android") == true)
	   return true;
       if (runtime.contains("Android") == true)
	   return true;
       if (runtime.contains("ANDROID") == true)
	   return true;

       return false;
       
   } // isAndroid()

   public boolean isSunJava()
   {
       String runtime;

       runtime = System.getProperty("java.runtime.name");
       if (runtime.contains("Java(TM)") == true)
	   return true;

       return false;

   } // isSunJava()

} /* class SocketOpts */
