/* XmpMessage.java

   COPYRIGHT 2004 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.xmp;

import java.util.Vector;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;  
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXException;  
import org.xml.sax.SAXParseException;  
import org.xml.sax.InputSource;
import java.io.IOException;
import org.w3c.dom.Document;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import java.io.StringReader;
import javax.swing.*;
import java.util.Iterator;

/** 
 * Base class serving as the internal representation of an XMP
 * message.  Instances contain all the fields and values of XMP
 * messages plus methods to convert to and from form suitable for wire
 * xfer.  XMP messages are transmitted in XML.  XMP messages contain a
 * type, identifier, error status, error index, and a list of
 * variables.  If the message type indicates an operation against a
 * MIB table, the XMP message will also contain a table name, table
 * MIB, table key, and the maximum number of rows the operation has
 * targeted.  XMP messages have borrowed heavily from the SNMP and
 * SMI.
 * @author Bobby Krupczak, rdk@krupczak.org
 * @version $Id: XmpMessage.java 15 2008-07-17 14:20:37Z rdk $
 * @see XmpVar
 * @see Xmp
 * @see XmpSession
 **/

public class XmpMessage {

  /* class variables and methods *********************** */

  /* instance variables **************************** */
  private int msgType;
  private long messageID;
  private int errorStatus;
  private int errorIndex;
  private int version;

  /* if table message, these fields are valid */
  private String tableMIB;
  private String tableName;
  private String tableKey;
  private int maxRows;

  /* message variables */
  private Vector varList;

  /* message encoded in XML */
  private String xmlEncoded;
  private boolean validDecode;
  private boolean validEncode;

  /* constructors  ***************************** */

  /** base constructor invoked by all other constructors **/
  public XmpMessage() 
  {
     varList = new Vector(0,10);
     messageID = System.currentTimeMillis() / 1000;
     errorStatus = Xmp.ERROR_NOERROR;
     errorIndex = 0;
     msgType = 0;
     validDecode = false;
     validEncode = false;
     version = Xmp.XMP_VERSION;

     /* by default, set tableInfo fields to empty */
     maxRows = 0;
     tableMIB = "";
     tableName = "";
     tableKey = "";

  } /* XmpMessage() */

  /** construct a message of given type **/
  public XmpMessage(int msgType)
  {
     this();

     /* sanity check msgType */
     if ((msgType < Xmp.MSGTYPE_RESPONSE) || (msgType > Xmp.MSGTYPE_TRAP)) {
         /* error or exception? */
        this.msgType = 0;
     }
     else 
        this.msgType = msgType;

  } /* XmpMessage(int msgType) */

  /** create a new message by deep-copying existing message **/
  public XmpMessage(XmpMessage toCopy)
  {
       Iterator anIterator;
       XmpVar toCopyVar;

       this.msgType = toCopy.getMsgType();
       this.messageID = toCopy.getMessageID();
       this.errorStatus = toCopy.getErrorStatus();
       this.errorIndex = toCopy.getErrorIndex();
       this.version = toCopy.getVersion();
       this.tableName = new String(toCopy.getTableName()); 
       this.tableMIB = new String(toCopy.getTableMIB());
       this.tableKey = new String(toCopy.getTableKey());
       this.maxRows = toCopy.getMaxRows();
       this.xmlEncoded = new String(toCopy.getXmlEncoding());
       this.validDecode = toCopy.isDecoded();
       this.validEncode = toCopy.isEncoded();

       // copy the variables via deep copy
       this.varList = new Vector(0,10);
 
       anIterator = toCopy.getVarIterator();
       while (anIterator.hasNext()) {
	 toCopyVar = (XmpVar)anIterator.next();
         this.addMIBVar(new XmpVar(toCopyVar));
       }
  }

  /* private methods **************************** */
  
  private XmpVar parseXmpVariable(Node n) 
  {
      int len,i,v,vlen;
      NodeList list,namelist;
      Node varname;
      Element varvalue;
      XmpVar aVar;

      if (n == null)
         return null;

      if (n.getNodeName().equals("variable") == false) {
         System.out.println("parseXmpVariable: not variable!");
         validDecode = false;
         return null;
      }
      aVar = new XmpVar();

      list = n.getChildNodes();
      len = list.getLength();
      i = 0;

      // varname <mib><object><key> 
      while (i < len) {
        if (list.item(i).getNodeName().equals("varname"))
	   break;
        i++;
      }
      if (i == len) {
	 System.out.println("Variable did not have varname");
         validDecode = false;
         return null;
      }
      varname = list.item(i);
      namelist = varname.getChildNodes();
      vlen = namelist.getLength();
      v = 0;
      while (v < vlen) {
	if (namelist.item(v).getNodeName().equals("mib"))
	   break;
        v++;
      }
      if (v == vlen) {
	 System.out.println("XMP variable does not have mib name");
         validDecode = false;
         return null;
      }
      if (namelist.item(v).getFirstChild() == null) {
	 System.out.println("XMP variable does not have mib name");
         validDecode = false;
         return null;
      }
      aVar.setMibName(namelist.item(v).getFirstChild().getNodeValue());

      v++;
      while (v < vlen) {
        if (namelist.item(v).getNodeName().equals("object"))
	   break;
        v++;
      }
      if (v == vlen) {
	 System.out.println("XMP variable does not have obj");
         validDecode = false;
	 return null;
      }
      if (namelist.item(v).getFirstChild() == null) {
	 System.out.println("XMP variable does not have obj");
         validDecode = false;
      }
      aVar.setObjName(namelist.item(v).getFirstChild().getNodeValue());

      // key could be null
      v++;
      while (v < vlen) {
        if (namelist.item(v).getNodeName().equals("key"))
	   break;
        v++;
      }
      if (v == vlen) {
	 System.out.println("XMP variable does not have key");
         validDecode = false;
	 return null;
      }
      if (namelist.item(v).getFirstChild() != null) {
	 aVar.setKey(namelist.item(v).getFirstChild().getNodeValue());
      }
      else {
	 aVar.setKey("");
      }

      // varvalue, vartype property, value
      i++;
      while (i < len) {
        if (list.item(i).getNodeName().equals("varvalue"))
	   break;
        i++;
      }
      if (i == len) {
	 System.out.println("Variable did not have varvalue");
         validDecode = false;
         return null;
      }
      varvalue = (Element)list.item(i);
      // vartype
      if (varvalue.hasAttributes() == false) {
	 System.out.println("Variable does not have type attribute");
         validDecode = false;
         return null;
      }
      aVar.setSyntax(varvalue.getAttribute("vartype"));

      // value itself
      if (varvalue.getFirstChild() != null) {
	 aVar.setValue(varvalue.getFirstChild().getNodeValue());
      }
      else {
	 // this should only be '' if vartype=displayString 
	 aVar.setValue("");
      }

      //aVar.dump(System.out);

      return aVar;

  } /* parseXmpVariable */

  /* public methods ***************************** */

  /* dump the internal message to stdout */
  public void dump() {
     int i;
     XmpVar aVar;

     System.out.println("PDU Dump ------------");

     if (validDecode != true) {
        System.out.println("Invalid XML or un-decoded");
     }
     if (validEncode != true) {
        System.out.println("Invalid XML or un-encoded");
     }

     System.out.println("Message type: "+Xmp.messageTypeToString(msgType));
     System.out.println("Message id: "+messageID);
     System.out.println("errorStatus: "+Xmp.errorStatusToString(errorStatus));
     System.out.println("errorIndex: "+errorIndex);

     // if table operation, then show tableInfo fields
     if (Xmp.isTableOperation(msgType)) {
        System.out.println("tableMIB: "+tableMIB);
        System.out.println("tableName: "+tableName);
        System.out.println("tableKey: "+tableKey);
        System.out.println("max-rows: "+maxRows);
     }

     // variables
     for (i=0; i<varList.size(); i++) {
         aVar = (XmpVar)varList.elementAt(i);
         aVar.dump(System.out);
     }

  } /* dump() */

  public void dump(JTextArea out) {
     int i;
     XmpVar aVar;

     if (validDecode != true) {
        out.append("Invalid XML or un-decoded\n");
	return;
     }

     out.append("Message type: "+Xmp.messageTypeToString(msgType)+"\n");
     out.append("Message id: "+messageID+"\n");
     out.append("errorStatus: "+Xmp.errorStatusToString(errorStatus)+"\n");
     out.append("errorIndex: "+errorIndex+"\n");

     // if table operation, then show tableInfo fields
     if (Xmp.isTableOperation(msgType)) {
        out.append("tableMIB: "+tableMIB+"\n");
        out.append("tableName: "+tableName+"\n");
        out.append("tableKey: "+tableKey+"\n");
        out.append("max-rows: "+maxRows+"\n");
     }

     // variables
     for (i=0; i<varList.size(); i++) {
         aVar = (XmpVar)varList.elementAt(i);
         aVar.dump(out);
     }

  } /* dump to text area */

  // take the current message and encode in XML into our buffer
  public int xmlEncodeMessage() {
     XmpVar[] vars;
     int i;
     StringBuffer buf;

     validEncode = false;

     // sanity check message first
     if (this.msgType == Xmp.MSGTYPE_INVALID)
        return -1;
     if (this.errorStatus == Xmp.ERROR_INVALID)
        return -1;

     // check table opts
     if ((this.msgType == Xmp.MSGTYPE_SELECTTABLEREQUEST) ||
        (this.msgType == Xmp.MSGTYPE_INSERTTABLEREQUEST) ||
        (this.msgType == Xmp.MSGTYPE_DELETETABLEREQUEST) ||
        (this.msgType == Xmp.MSGTYPE_UPDATETABLEREQUEST)) {
        if ((tableMIB == null) || (tableName == null)
	    || (tableKey == null))
	   return -1;
        // sanity check maxRows? 
     }

     // check vars 
     vars = this.getMIBVars();
     for (i=0; i<this.getNumberVars(); i++) {
         if (vars[i].xmpSyntax == Xmp.SYNTAX_INVALID)
	    return -1;
     }
   
     // now, start encoding
     buf = new StringBuffer();
     buf.append("<?xml version=\"1.0\" encoding='US-ASCII' ?>\n");
     //buf.append("<!DOCTYPE xmp>\n");
     //buf.append("<xmp message-id=\""+messageID+"\">\n");

     buf.append("<xmp xmlns=\"http://xmlns.krupczak.org/xsd/xmp-1.0\" ");
     buf.append("message-id=\""+messageID+"\">\n");

     buf.append("<"+Xmp.messageTypeToString(msgType)+">\n");
     buf.append("<error-status>"+Xmp.errorStatusToString(errorStatus)+
                "</error-status>\n");
     buf.append("<error-index>"+errorIndex+"</error-index>\n");
     
     // if its a table operation, put in <tableInfo>
     if ((msgType >= Xmp.MSGTYPE_SELECTTABLEREQUEST) && 
	 (msgType <= Xmp.MSGTYPE_UPDATETABLEREQUEST)) {
         buf.append("<tableInfo>\n");
         buf.append("<mib>"+tableMIB+"</mib>\n");
         buf.append("<object>"+tableName+"</object>\n");
         buf.append("<key>"+tableKey+"</key>\n");
         buf.append("<max-rows>"+maxRows+"</max-rows>\n");
         buf.append("</tableInfo>\n");
     }

     buf.append("<variableList>\n");
     for (i=0; i<this.getNumberVars(); i++) {
         buf.append("<variable>\n");

         // name
         buf.append("<varname><mib>"+vars[i].mibName+"</mib>");
         buf.append("<object>"+vars[i].objName+"</object>");
         if (vars[i].key == null)
	    buf.append("<key/>");
         else
            buf.append("<key>"+vars[i].key+"</key></varname>\n");

         // value
         if (vars[i].xmpSyntax == Xmp.SYNTAX_NULLSYNTAX)
	     buf.append("<varvalue vartype=\"nullSyntax\"/>\n");
         else 
             buf.append("<varvalue vartype=\""+
                        Xmp.syntaxToString(vars[i].xmpSyntax)+"\">"+
                        vars[i].valueStr+"</varvalue>\n");

         buf.append("</variable>\n");
     }
     buf.append("</variableList>\n");
     buf.append("</"+Xmp.messageTypeToString(msgType)+">\n");
     buf.append("</xmp>\n");

     this.xmlEncoded = new String(buf);

     validEncode = true;

     return this.xmlEncoded.length();
  }

  public int xmlDecodeMessage(byte[] input) 
  {
      DocumentBuilderFactory db;
      Document xmldoc;
      DocumentBuilder b;
      Node n;
      NodeList nlist,tinfolist,varlist;
      int i, nlistlen,tinfolistlen,varlistlen,t;
      XmpVar aVar;

      // copy/convert the xml message into our internal buffer
      xmlEncoded = new String(input);

      // parse!
      db = DocumentBuilderFactory.newInstance();

      try {
        b = db.newDocumentBuilder();

        xmldoc = b.parse(new InputSource(new StringReader(xmlEncoded)));

        // doctype -- (DTD) name immediately following DOCTYPE keyword
        // should be 'xmp'
        // or, we have a schema namespace 
        if (xmldoc.getDoctype() == null) {
	    //validDecode = false;
	    //return -6;
            // this means we have no DTD statement because we've
            // switched to an XML schema definition
            //System.out.println("xmlDecodeMessage: no DTD, no biggie");
        }
        else {
           if (xmldoc.getDoctype().getName().equals("xmp") == false) {
              validDecode = false;
   	      return -7;
           }
        }
        if (xmldoc.getDocumentElement() == null) {
            System.out.println("xmlDecodeMessage: no xmp element");
	    validDecode = false;
	    return -6;
        }

        // doc element -- root element of the document; for us, it
        // should be <xmp namespace message-id="X">
        if (xmldoc.getDocumentElement().getTagName().equals("xmp")==false) {
           validDecode = false;
	   return -8;
        }
        if (xmldoc.getDocumentElement().hasAttributes() == false) {
	   // needs to have a message-id
           validDecode = false;
	   return -9;
        }
        if (xmldoc.getDocumentElement().getAttribute("message-id") == null) {
	   // has attributes but apparently not message-id
           validDecode = false;
	   return -10;
        }
        messageID = Long.valueOf(xmldoc.getDocumentElement().getAttribute("message-id")).longValue();

        // message type is only element child of doc element
        n = (Node)xmldoc.getDocumentElement();
        if ((nlist = n.getChildNodes()) == null) { 
           validDecode = false;
	   return -11;
        }
        for (i=0; i<nlist.getLength(); i++) {
            if (nlist.item(i).getNodeType() == Node.ELEMENT_NODE) {
	       n = nlist.item(i);
	       msgType = Xmp.stringToMessageType(n.getNodeName());
            }
        }
        if (msgType == Xmp.MSGTYPE_INVALID) {
           validDecode = false;
	   return -12;
        }

        // n is now <messagetype> node; error-status, error-index,
        // optional-tableInfo, and variableList are all child nodes
        if ((nlist = n.getChildNodes()) == null) {
           validDecode = false;
	   return -13;
        }

        // error-status
        i = 0;
        nlistlen = nlist.getLength();
        while (i < nlistlen) {
	  if (nlist.item(i).getNodeName().equals("error-status"))
	     break;
	  i++;
        }
        if (i == nlistlen) { validDecode = false; return -1; }
        if (nlist.item(i).hasChildNodes() == false) { 
           validDecode = false; return -14; 
        }
        errorStatus = Xmp.stringToErrorStatus(nlist.item(i).getFirstChild().getNodeValue());
        if (errorStatus == Xmp.ERROR_INVALID) { validDecode = false;return -15;}

        // error-index
        i++;
        while (i < nlistlen) {
          if (nlist.item(i).getNodeName().equals("error-index"))
	     break;
	  i++;
        }
        if (i == nlistlen) { validDecode = false; return -16; }
        if (nlist.item(i).hasChildNodes() == false) { validDecode = false; return -1; }
        errorIndex = Integer.valueOf(nlist.item(i).getFirstChild().getNodeValue()).intValue();

        // tableInfo if messagetype is table operation
        if (Xmp.isTableOperation(msgType) == true) {
           i++;
           while (i < nlistlen) {
	     if (nlist.item(i).getNodeName().equals("tableInfo"))
		break;
	     i++;
           }
           if (i == nlistlen) { validDecode = false; return -17; }
           if (nlist.item(i).hasChildNodes() == false) { validDecode = false; return -1; }
           tinfolist = nlist.item(i).getChildNodes();
           tinfolistlen = tinfolist.getLength();

           // parse <mib>
           t = 0;
           while (t < tinfolistlen) {
             if (tinfolist.item(t).getNodeName().equals("mib"))  
	        break;
	     t++;
           }
           if (t == tinfolistlen) { validDecode = false; return -18; }
           if (tinfolist.item(t).hasChildNodes() == false) { validDecode = false; return -1; }
           tableMIB = tinfolist.item(t).getFirstChild().getNodeValue();

           // parse <object>
           t++;
           while (t < tinfolistlen) {
             if (tinfolist.item(t).getNodeName().equals("object"))  
	        break;
	     t++;
           }
           if (t == tinfolistlen) { validDecode = false; return -19; }
           if (tinfolist.item(t).hasChildNodes() == false) { validDecode = false; return -1; }
           tableName = tinfolist.item(t).getFirstChild().getNodeValue();

	   // parse <key> -- must be present
           t++;
           while (t < tinfolistlen) {
             if (tinfolist.item(t).getNodeName().equals("key"))  
	        break;
	     t++;
           }
           if (t == tinfolistlen) { validDecode = false; return -20; }
           if (tinfolist.item(t).hasChildNodes() == false) { validDecode = false; return -1; }
           tableKey = tinfolist.item(t).getFirstChild().getNodeValue();

	   // parse <max-rows> -- must be present
           t++;
           while (t < tinfolistlen) {
             if (tinfolist.item(t).getNodeName().equals("max-rows"))  
	        break;
	     t++;
           }
           if (t == tinfolistlen) { validDecode = false; return -21; }
           if (tinfolist.item(t).hasChildNodes() == false) { validDecode = false; return -1; }
           maxRows = Integer.valueOf(tinfolist.item(t).getFirstChild().getNodeValue()).intValue();

        } /* parse tableInfo */

        // variableList
        i++;
        while (i < nlistlen) {
          if (nlist.item(i).getNodeName().equals("variableList"))
	     break;
	  i++;
        }
        if (i == nlistlen) { validDecode = false; return -22; }
        if (nlist.item(i).hasChildNodes() == false) { validDecode = false; return -1; }
        varlist = nlist.item(i).getChildNodes();    
        varlistlen = varlist.getLength();

        // variables
	i = 0;
        while (i < varlistlen) {
          if (varlist.item(i).getNodeName().equals("variable")) {
	     aVar = parseXmpVariable(varlist.item(i));
             if (aVar == null) {
                validDecode = false;
	        return -23;
             }
             varList.add(aVar);
          }
	  i++;
        }

      } catch (SAXParseException spe) { 
          System.out.println("Spe "+spe.getMessage()); 
          System.out.println("XML message is");
          System.out.println(xmlEncoded);
          validDecode = false; 
          return -2;
      }	catch (SAXException sxe) { 
          validDecode = false; 
          return -3; 
      } catch (ParserConfigurationException pce) { 
          validDecode = false; 
          return -4;
      } catch (IOException e) { 
          validDecode = false; 
          return -5;
      }
     
      validDecode = true;

      return 1;
  }

  public int getVersion() { return version; }
  public int getMsgType() { return msgType; }
  public String getMsgTypeStr() { return Xmp.messageTypeToString(msgType); };
  public long getMessageID() { return messageID; }
  public int getErrorStatus() { return errorStatus; }
  public int getErrorIndex() { return errorIndex; }
  public String getTableName() { return tableName; }
  public String getTableMIB() { return tableMIB; }
  public String getTableKey() { return tableKey; }
  public int getMaxRows() { return maxRows; }
  public String getXmlEncoding() { return xmlEncoded; }

  public boolean isEncoded() { return validEncode; }
  public boolean isDecoded() { return validDecode; }
  public int setDecoded() { validDecode = true; return 1; }
  public int setEncoded() { validEncode = true; return 1; }
  public void setUnencoded() { validEncode = false; return; }

  public int setMsgType(int msgType) { 
     /* sanity check msgType */
     if ((msgType < Xmp.MSGTYPE_RESPONSE) || (msgType > Xmp.MSGTYPE_TRAP)) {
         /* error or exception? */
        this.msgType = 0;
        return -1;
     }
     else 
        this.msgType = msgType;

     return 0;
  } /* setMsgType */

  public int setMessageID(long messageID) 
  { 
     this.messageID = messageID;  
     return 0;
  }

  public int setErrorStatus(int errorStatus) 
  { 
     if ((errorStatus < 0) || (errorStatus > Xmp.ERROR_INCONSISTENTNAME)) {
        return -1;
     }
     this.errorStatus = errorStatus;

     return 0; 
  }

  public int setErrorIndex(int errorIndex) 
  { 
     if (errorIndex < 0) {
        return -1;
     }
     else 
        this.errorIndex = errorIndex;

     return 0; 
  }

  public int setTableName(String tableName) 
  { 
     this.tableName = tableName;
     return 0; 
  }
  public int setTableMIB(String tableMIB) 
  { 
     this.tableMIB = tableMIB;
     return 0; 
  }
  public int setTableKey(String tableKey) 
  { 
     this.tableKey = tableKey;
     return 0; 
  }
  public int setMaxRows(int maxRows) 
  { 
     if (maxRows < 0)
	return -1;
     this.maxRows = maxRows;
     return 0; 
  }
  public int setTableOperands(String tableMIB, String tableName, 
                              String tableKey, int maxRows)
  {
     this.setTableName(tableName);
     this.setTableMIB(tableMIB);
     this.setTableKey(tableKey);
     this.setMaxRows(maxRows);
     return 0;
  }


  /* add/get/set MIB variables */
 
  public Iterator getVarIterator() { return varList.iterator(); }

  public int addMIBVar(XmpVar var) { 
     if (var == null)
        return -1;        
     varList.add(var);
     return 1; 
  }

  public XmpVar[] getMIBVars() { 
     return (XmpVar[])varList.toArray(new XmpVar[0]);
  }

  public int setMIBVars(XmpVar[] vars) 
  { 
     int i;

     varList = new Vector(); 
     for (i=0; i<vars.length; i++)
         varList.add(vars[i]);

     return 1;
  }

  public XmpVar getMIBVar(int index) {
     XmpVar[] anArray;

     // index 0..size-1 
     if (index >= varList.size())
	return null;

     anArray = (XmpVar[])varList.toArray(new XmpVar[0]);

     return anArray[index];
  }   

  public int getNumberVars() { return varList.size(); }

} /* class XmpMessage */
