/* Trapreceiver.java

   COPYRIGHT 2013 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.xmp;

import java.net.*;
import java.io.*;
import java.security.*;
import java.security.cert.*;
import javax.net.ssl.*;

/**
 *  XMP Trap receiver 
 *  @author Bobby Krupczak, rdk@krupczak.org
 *  @version $Id$
 *  @see XmpSession
 *  @see XmpMessage
 *  @see XmpListener
 *  @see Xmp
 **/

public class Trapreceiver implements Runnable {

    private static void usage(String argv[]) {

	System.out.println("Usage: java Trapreceiver addr [port]");
        System.exit(-1);

    } /* usage */

    public XmpSession aSession;

    public Trapreceiver(XmpSession aSession) {
       this.aSession = aSession;
    }

    // send a generic response for non-Trap messages
    private int sendGenericResponse(XmpMessage aRequest)
    {
	XmpMessage aReply;
        int ret;

        XmpVar[] vars = new XmpVar[] {
           new XmpVar("core","sysDescr","","Java Trapreceiver tool",
                       Xmp.SYNTAX_DISPLAYSTRING),
           new XmpVar("core","sysObjectID","","1.3.6.1.4.1.16050",
                      Xmp.SYNTAX_DISPLAYSTRING),
        };

        aReply = new XmpMessage(Xmp.MSGTYPE_RESPONSE);
        aReply.setDecoded();

        aReply.setMIBVars(vars);
        aReply.setMessageID(aRequest.getMessageID());

        // set isEncoded false to force re encoding
        aReply.setUnencoded();

        if ((ret = aSession.sendMessage(aReply)) < 0) {
            System.out.println("Trapreceiver: send failed, "+ret);
        }

        return ret;
    }

    // acknowledge the trap via response
    private int sendTrapResponse(XmpMessage aTrap)
    {
	int ret;
        XmpMessage aReply = new XmpMessage(aTrap);

        System.out.println("Responding to Trap PDU");

        aReply.setMsgType(Xmp.MSGTYPE_RESPONSE);
        // set isEncoded false to force re encoding
        aReply.setUnencoded();
        
        if ((ret = aSession.sendMessage(aReply)) < 0) {
            System.out.println("Trapreceiver: send failed, "+ret);
        }

        return ret;
    }

    // acknowledge the information via response
    private int sendInformationResponse(XmpMessage anInfo)
    {
	int ret;
        XmpMessage aReply = new XmpMessage(anInfo);

        System.out.println("Responding to Information PDU");

        aReply.setMsgType(Xmp.MSGTYPE_RESPONSE); 
        // set isEncoded false to force re encoding
        aReply.setUnencoded();

        if ((ret = aSession.sendMessage(aReply)) < 0) {
            System.out.println("Trapreceiver: send failed, "+ret);
        }

        return ret;
    }
 
    public void run() 
    {
         System.out.println("Trapreceiver accepted connection via "+
                             aSession.getSessionProto()+" over IPv"+
                             aSession.getIpVersion()+" from "+
                             aSession.getTargetAddr()+","+
                             aSession.getTargetPort());

         // wait for messages, process, and reply 
         XmpMessage aRequest;
         int ret;

         while (true) {
         
 	   aRequest = aSession.recvMessage();
           if (aRequest == null) {
              System.out.println("Trapreceiver: recv error "+
                                 aSession.getErrorStatus());         
              break;
           }

           switch (aRequest.getMsgType()) {
	     case Xmp.MSGTYPE_TRAP:
                 sendTrapResponse(aRequest);
                 break;
	     case Xmp.MSGTYPE_INFORMATION: 
                 sendInformationResponse(aRequest);
                 break;
	     default:
		 sendGenericResponse(aRequest);
           }

	 } /* while */

         System.out.println("Trapreceiver: run finishing");

    } /* run */

    public static void main(String argv[]) 
    {
        int i,portNumber;
        XmpListener listener;
        SocketOpts sockopts;
        XmpSession clientSession;
       
        /* java and program name are not part of argv */
        if (argv.length < 1)
	    usage(argv);

        sockopts = new SocketOpts();

        if (argv.length > 1) 
	    portNumber = Integer.valueOf(argv[1]).intValue();
        else {
            // use XMP_PORT+1 for now since we are probably running an
            // agent on our local system
            portNumber = Xmp.XMP_PORT+1;
        }

        System.out.println("XMP Trap receiver going to listen on port "+
                            portNumber);

        listener = new XmpListener(sockopts,"*",portNumber);

        while (true) {

          if ((clientSession = listener.listenForConnection()) == null) {
  	      System.out.println("Trapreceiver session accept error");
              break;
          }
          else {
	      new Thread(new Trapreceiver(clientSession)).start();
          }

        } /* while true, listen for connection requests */

    } /* main */

} /* class Trapreceiver */
