/*
 * Decompiled with CFR 0.152.
 */
package org.opennms.core.ipc.rpc.kafka;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.JmxReporter;
import com.codahale.metrics.Meter;
import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Strings;
import com.google.common.math.IntMath;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.swrve.ratelimitedlogger.RateLimitedLog;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.opentracing.propagation.Format;
import java.math.RoundingMode;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.joda.time.Duration;
import org.opennms.core.camel.JmsQueueNameFactory;
import org.opennms.core.ipc.common.kafka.KafkaRpcConstants;
import org.opennms.core.ipc.common.kafka.OnmsKafkaConfigProvider;
import org.opennms.core.ipc.rpc.kafka.ResponseCallback;
import org.opennms.core.ipc.rpc.kafka.model.RpcMessageProtos;
import org.opennms.core.logging.Logging;
import org.opennms.core.rpc.api.RemoteExecutionException;
import org.opennms.core.rpc.api.RequestTimedOutException;
import org.opennms.core.rpc.api.RpcClient;
import org.opennms.core.rpc.api.RpcClientFactory;
import org.opennms.core.rpc.api.RpcModule;
import org.opennms.core.rpc.api.RpcRequest;
import org.opennms.core.rpc.api.RpcResponse;
import org.opennms.core.tracing.api.TracerRegistry;
import org.opennms.core.tracing.util.TracingInfoCarrier;
import org.opennms.core.utils.SystemInfoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

public class KafkaRpcClientFactory
implements RpcClientFactory {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaRpcClientFactory.class);
    private static final RateLimitedLog RATE_LIMITED_LOG = RateLimitedLog.withRateLimit((Logger)LOG).maxRate(5).every(Duration.standardSeconds((long)30L)).build();
    private String location;
    private KafkaProducer<String, byte[]> producer;
    private final Properties kafkaConfig = new Properties();
    private final ThreadFactory consumerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-client-kafka-consumer-%d").build();
    private final ThreadFactory responseHandlerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-client-response-handler-%d").build();
    private final ThreadFactory timerThreadFactory = new ThreadFactoryBuilder().setNameFormat("rpc-client-timeout-tracker-%d").build();
    private final ExecutorService kafkaConsumerExecutor = Executors.newSingleThreadExecutor(this.consumerThreadFactory);
    private final ExecutorService timerExecutor = Executors.newSingleThreadExecutor(this.timerThreadFactory);
    private final ExecutorService responseHandlerExecutor = Executors.newCachedThreadPool(this.responseHandlerThreadFactory);
    private final Map<String, ResponseCallback> rpcResponseMap = new ConcurrentHashMap<String, ResponseCallback>();
    private KafkaConsumerRunner kafkaConsumerRunner;
    private DelayQueue<ResponseCallback> delayQueue = new DelayQueue();
    private Map<String, ByteString> messageCache = new ConcurrentHashMap<String, ByteString>();
    private MetricRegistry metrics;
    private Map<String, Integer> currentChunkCache = new ConcurrentHashMap<String, Integer>();
    private JmxReporter metricsReporter = null;
    @Autowired
    private TracerRegistry tracerRegistry;
    private Tracer tracer;

    public <S extends RpcRequest, T extends RpcResponse> RpcClient<S, T> getClient(final RpcModule<S, T> module) {
        return new RpcClient<S, T>(){

            public CompletableFuture<T> execute(S request) {
                if (request.getLocation() == null || request.getLocation().equals(KafkaRpcClientFactory.this.location)) {
                    return module.execute(request);
                }
                Span span = KafkaRpcClientFactory.this.tracer.buildSpan(module.getId()).start();
                JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-request", module.getId(), request.getLocation());
                String requestTopic = topicNameFactory.getName();
                String marshalRequest = module.marshalRequest(request);
                String rpcId = UUID.randomUUID().toString();
                Long ttl = request.getTimeToLiveMs();
                ttl = ttl != null && ttl > 0L ? ttl : KafkaRpcConstants.DEFAULT_TTL;
                long expirationTime = System.currentTimeMillis() + ttl;
                CompletableFuture future = new CompletableFuture();
                Map loggingContext = Logging.getCopyOfContextMap();
                ResponseHandler responseHandler = new ResponseHandler(future, module, rpcId, expirationTime, loggingContext, request.getLocation(), span);
                KafkaRpcClientFactory.this.delayQueue.offer(responseHandler);
                KafkaRpcClientFactory.this.rpcResponseMap.put(rpcId, responseHandler);
                KafkaRpcClientFactory.this.kafkaConsumerRunner.startConsumingForModule(module.getId());
                byte[] messageInBytes = marshalRequest.getBytes();
                int totalChunks = IntMath.divide((int)messageInBytes.length, (int)KafkaRpcConstants.MAX_BUFFER_SIZE, (RoundingMode)RoundingMode.UP);
                RpcMessageProtos.RpcMessage.Builder builder = RpcMessageProtos.RpcMessage.newBuilder().setRpcId(rpcId).setSystemId(request.getSystemId() == null ? "" : request.getSystemId()).setExpirationTime(expirationTime);
                for (int chunk = 0; chunk < totalChunks; ++chunk) {
                    int bufferSize = KafkaRpcConstants.getBufferSize((int)messageInBytes.length, (int)KafkaRpcConstants.MAX_BUFFER_SIZE, (int)chunk);
                    ByteString byteString = ByteString.copyFrom((byte[])messageInBytes, (int)(chunk * KafkaRpcConstants.MAX_BUFFER_SIZE), (int)bufferSize);
                    int chunkNum = chunk;
                    this.addTracingInfo((RpcRequest)request, span, builder);
                    RpcMessageProtos.RpcMessage rpcMessage = builder.setRpcContent(byteString).setCurrentChunkNumber(chunk).setTotalChunks(totalChunks).build();
                    Callback sendCallback = (recordMetadata, e) -> {
                        if (e != null) {
                            RATE_LIMITED_LOG.error(" RPC request {} with id {} couldn't be sent to Kafka", new Object[]{request, rpcId, e});
                            future.completeExceptionally(e);
                        } else if (LOG.isTraceEnabled()) {
                            LOG.trace("RPC Request {} with id {} chunk {} sent to minion at location {}", new Object[]{request, rpcId, chunkNum, request.getLocation()});
                        }
                    };
                    if (request.getSystemId() != null) {
                        List partitionInfo = KafkaRpcClientFactory.this.producer.partitionsFor(requestTopic);
                        partitionInfo.forEach(partition -> {
                            ProducerRecord record = new ProducerRecord(requestTopic, Integer.valueOf(partition.partition()), (Object)rpcId, (Object)rpcMessage.toByteArray());
                            KafkaRpcClientFactory.this.producer.send(record, sendCallback);
                        });
                        continue;
                    }
                    ProducerRecord record = new ProducerRecord(requestTopic, (Object)rpcId, (Object)rpcMessage.toByteArray());
                    KafkaRpcClientFactory.this.producer.send(record, sendCallback);
                }
                this.addMetrics((RpcRequest)request, messageInBytes.length);
                return future;
            }

            private void addMetrics(RpcRequest request, int messageLen) {
                Meter requestSentMeter = KafkaRpcClientFactory.this.getMetrics().meter(MetricRegistry.name((String)request.getLocation(), (String[])new String[]{module.getId(), "requestSent"}));
                requestSentMeter.mark();
                Histogram rpcRequestSize = KafkaRpcClientFactory.this.getMetrics().histogram(MetricRegistry.name((String)request.getLocation(), (String[])new String[]{module.getId(), "requestSize"}));
                rpcRequestSize.update(messageLen);
            }

            private void addTracingInfo(RpcRequest request, Span span, RpcMessageProtos.RpcMessage.Builder builder) {
                span.setTag("location", request.getLocation());
                if (request.getSystemId() != null) {
                    span.setTag("systemId", request.getSystemId());
                }
                request.getTracingInfo().forEach((arg_0, arg_1) -> ((Span)span).setTag(arg_0, arg_1));
                TracingInfoCarrier tracingInfoCarrier = new TracingInfoCarrier();
                KafkaRpcClientFactory.this.tracer.inject(span.context(), Format.Builtin.TEXT_MAP, (Object)tracingInfoCarrier);
                tracingInfoCarrier.getTracingInfoMap().forEach((key, value) -> this.buildTracingInfo(builder, (String)key, (String)value));
                request.getTracingInfo().forEach((key, value) -> this.buildTracingInfo(builder, (String)key, (String)value));
            }

            private void buildTracingInfo(RpcMessageProtos.RpcMessage.Builder builder, String key, String value) {
                if (!Strings.isNullOrEmpty((String)key) && !Strings.isNullOrEmpty((String)value)) {
                    RpcMessageProtos.TracingInfo tracingInfo = RpcMessageProtos.TracingInfo.newBuilder().setKey(key).setValue(value).build();
                    builder.addTracingInfo(tracingInfo);
                }
            }
        };
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public void setTracerRegistry(TracerRegistry tracerRegistry) {
        this.tracerRegistry = tracerRegistry;
    }

    public TracerRegistry getTracerRegistry() {
        return this.tracerRegistry;
    }

    public MetricRegistry getMetrics() {
        if (this.metrics == null) {
            this.metrics = new MetricRegistry();
        }
        return this.metrics;
    }

    public void setMetrics(MetricRegistry metrics) {
        this.metrics = metrics;
    }

    public void start() {
        try (Logging.MDCCloseable mdc = Logging.withPrefixCloseable((String)"ipc");){
            this.kafkaConfig.clear();
            this.kafkaConfig.put("group.id", SystemInfoUtils.getInstanceId());
            this.kafkaConfig.put("auto.commit.interval.ms", "1000");
            this.kafkaConfig.put("key.deserializer", StringDeserializer.class.getCanonicalName());
            this.kafkaConfig.put("value.deserializer", ByteArrayDeserializer.class.getCanonicalName());
            this.kafkaConfig.put("key.serializer", StringSerializer.class.getCanonicalName());
            this.kafkaConfig.put("value.serializer", ByteArraySerializer.class.getCanonicalName());
            OnmsKafkaConfigProvider kafkaConfigProvider = new OnmsKafkaConfigProvider("org.opennms.core.ipc.rpc.kafka.");
            this.kafkaConfig.putAll((Map<?, ?>)kafkaConfigProvider.getProperties());
            this.producer = new KafkaProducer(this.kafkaConfig);
            LOG.info("initializing the Kafka producer with: {}", (Object)this.kafkaConfig);
            this.startKafkaConsumer();
            this.metricsReporter = JmxReporter.forRegistry((MetricRegistry)this.getMetrics()).inDomain("org.opennms.core.ipc.rpc").build();
            this.metricsReporter.start();
            this.tracerRegistry.init(SystemInfoUtils.getInstanceId());
            this.tracer = this.tracerRegistry.getTracer();
            LOG.info("started  kafka consumer with : {}", (Object)this.kafkaConfig);
            this.timerExecutor.execute(() -> {
                while (true) {
                    try {
                        while (true) {
                            ResponseCallback responseCb;
                            if ((responseCb = (ResponseCallback)this.delayQueue.take()).isProcessed()) {
                                continue;
                            }
                            LOG.warn("RPC request with id {} timedout ", (Object)responseCb.getRpcId());
                            this.responseHandlerExecutor.execute(() -> responseCb.sendResponse(null));
                        }
                    }
                    catch (InterruptedException e) {
                        LOG.info("interrupted while waiting for an element from delayQueue", (Throwable)e);
                    }
                    catch (Exception e) {
                        LOG.warn("error while sending response from timeout handler", (Throwable)e);
                        continue;
                    }
                    break;
                }
            });
            LOG.info("started timeout tracker");
        }
    }

    private synchronized void startKafkaConsumer() {
        KafkaConsumer kafkaConsumer = new KafkaConsumer(this.kafkaConfig);
        this.kafkaConsumerRunner = new KafkaConsumerRunner(kafkaConsumer);
        this.kafkaConsumerExecutor.execute(this.kafkaConsumerRunner);
    }

    public void stop() {
        LOG.info("stop kafka consumer runner");
        if (this.metricsReporter != null) {
            this.metricsReporter.close();
        }
        this.kafkaConsumerRunner.stop();
        this.kafkaConsumerExecutor.shutdown();
        this.timerExecutor.shutdown();
        this.responseHandlerExecutor.shutdown();
    }

    private class KafkaConsumerRunner
    implements Runnable {
        private final KafkaConsumer<String, byte[]> consumer;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        private final Set<String> moduleIdsForTopics = new HashSet<String>();
        private final Set<String> topics = new HashSet<String>();
        private final AtomicBoolean topicAdded = new AtomicBoolean(false);
        private final CountDownLatch firstTopicAdded = new CountDownLatch(1);

        private KafkaConsumerRunner(KafkaConsumer<String, byte[]> consumer) {
            this.consumer = consumer;
        }

        @Override
        public void run() {
            Logging.putPrefix((String)"ipc");
            if (this.topics.isEmpty()) {
                this.waitTillFirstTopicIsAdded();
                LOG.info("First topic is added, consumer will be started.");
            }
            while (!this.closed.get()) {
                try {
                    this.subscribeToTopics();
                    ConsumerRecords records = this.consumer.poll(java.time.Duration.ofMillis(Long.MAX_VALUE));
                    for (ConsumerRecord record : records) {
                        ResponseCallback responseCb = (ResponseCallback)KafkaRpcClientFactory.this.rpcResponseMap.get(record.key());
                        if (responseCb != null) {
                            RpcMessageProtos.RpcMessage rpcMessage = RpcMessageProtos.RpcMessage.parseFrom((byte[])record.value());
                            ByteString rpcContent = rpcMessage.getRpcContent();
                            String rpcId = rpcMessage.getRpcId();
                            if (rpcMessage.getTotalChunks() > 1) {
                                boolean allChunksReceived = this.handleChunks(rpcMessage);
                                if (!allChunksReceived) continue;
                                rpcContent = (ByteString)KafkaRpcClientFactory.this.messageCache.get(rpcId);
                            }
                            if (LOG.isTraceEnabled()) {
                                LOG.trace("Received RPC response for id {}", (Object)rpcMessage.getRpcId());
                            }
                            String rpcMessageContent = rpcContent.toStringUtf8();
                            KafkaRpcClientFactory.this.responseHandlerExecutor.execute(() -> responseCb.sendResponse(rpcMessageContent));
                            KafkaRpcClientFactory.this.rpcResponseMap.remove(rpcId);
                            KafkaRpcClientFactory.this.messageCache.remove(rpcId);
                            KafkaRpcClientFactory.this.currentChunkCache.remove(rpcId);
                            continue;
                        }
                        LOG.debug("Received a response for request with ID:{}, but no outstanding request was found with this id.The request may have timed out or the response may be a duplicate.", record.key());
                    }
                }
                catch (InvalidProtocolBufferException e) {
                    LOG.error("error while parsing response", (Throwable)e);
                }
                catch (WakeupException e) {
                    LOG.info("consumer got wakeup exception, closed = {} ", (Object)this.closed.get(), (Object)e);
                }
                catch (Throwable e) {
                    LOG.error("Unexpected error in kafka consumer.", e);
                }
            }
            this.consumer.close();
        }

        public void stop() {
            this.closed.set(true);
            this.consumer.wakeup();
        }

        private boolean handleChunks(RpcMessageProtos.RpcMessage rpcMessage) {
            String rpcId = rpcMessage.getRpcId();
            KafkaRpcClientFactory.this.currentChunkCache.putIfAbsent(rpcId, 0);
            Integer chunkNumber = (Integer)KafkaRpcClientFactory.this.currentChunkCache.get(rpcId);
            if (chunkNumber.intValue() != rpcMessage.getCurrentChunkNumber()) {
                LOG.debug("Expected chunk = {} but got chunk = {}, ignoring.", (Object)chunkNumber, (Object)rpcMessage.getCurrentChunkNumber());
                return false;
            }
            ByteString byteString = (ByteString)KafkaRpcClientFactory.this.messageCache.get(rpcId);
            if (byteString != null) {
                KafkaRpcClientFactory.this.messageCache.put(rpcId, byteString.concat(rpcMessage.getRpcContent()));
            } else {
                KafkaRpcClientFactory.this.messageCache.put(rpcId, rpcMessage.getRpcContent());
            }
            chunkNumber = chunkNumber + 1;
            KafkaRpcClientFactory.this.currentChunkCache.put(rpcId, chunkNumber);
            return rpcMessage.getTotalChunks() == chunkNumber.intValue();
        }

        private void waitTillFirstTopicIsAdded() {
            while (!this.topicAdded.get()) {
                try {
                    this.firstTopicAdded.await(1L, TimeUnit.SECONDS);
                }
                catch (InterruptedException e) {
                    LOG.info("Interrupted before first topic was added. Terminating Kafka RPC consumer thread.", (Throwable)e);
                    return;
                }
                catch (Throwable e) {
                    LOG.error("Unknown exception before first topic is added. Terminating Kafka RPC consumer thread.", e);
                    return;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void subscribeToTopics() {
            if (this.topicAdded.get()) {
                Set<String> set = this.topics;
                synchronized (set) {
                    LOG.info("Subscribing Kafka RPC consumer to topics named: {}", this.topics);
                    this.consumer.subscribe(this.topics);
                    this.topicAdded.set(false);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void startConsumingForModule(String moduleId) {
            if (this.moduleIdsForTopics.contains(moduleId)) {
                return;
            }
            this.moduleIdsForTopics.add(moduleId);
            JmsQueueNameFactory topicNameFactory = new JmsQueueNameFactory("rpc-response", moduleId);
            Set<String> set = this.topics;
            synchronized (set) {
                if (this.topics.add(topicNameFactory.getName())) {
                    this.topicAdded.set(true);
                    this.firstTopicAdded.countDown();
                    this.consumer.wakeup();
                }
            }
        }
    }

    private class ResponseHandler<S extends RpcRequest, T extends RpcResponse>
    implements ResponseCallback {
        private final CompletableFuture<T> responseFuture;
        private final RpcModule<S, T> rpcModule;
        private final long expirationTime;
        private final String rpcId;
        private Map<String, String> loggingContext;
        private boolean isProcessed = false;
        private final String location;
        private Span span;
        private final Long requestCreationTime;
        private final Histogram rpcDuration;
        private final Meter failedMeter;
        private final Histogram responseSize;

        private ResponseHandler(CompletableFuture<T> responseFuture, RpcModule<S, T> rpcModule, String rpcId, long timeout, Map<String, String> loggingContext, String location, Span span) {
            this.responseFuture = responseFuture;
            this.rpcModule = rpcModule;
            this.expirationTime = timeout;
            this.rpcId = rpcId;
            this.loggingContext = loggingContext;
            this.span = span;
            this.location = location;
            this.requestCreationTime = System.currentTimeMillis();
            this.rpcDuration = KafkaRpcClientFactory.this.getMetrics().histogram(MetricRegistry.name((String)location, (String[])new String[]{rpcModule.getId(), "duration"}));
            this.failedMeter = KafkaRpcClientFactory.this.getMetrics().meter(MetricRegistry.name((String)location, (String[])new String[]{rpcModule.getId(), "requestFailed"}));
            this.responseSize = KafkaRpcClientFactory.this.getMetrics().histogram(MetricRegistry.name((String)location, (String[])new String[]{rpcModule.getId(), "responseSize"}));
        }

        @Override
        public void sendResponse(String message) {
            try (Logging.MDCCloseable mdc = Logging.withContextMapCloseable(this.loggingContext);){
                if (message != null) {
                    RpcResponse response = this.rpcModule.unmarshalResponse(message);
                    if (response.getErrorMessage() != null) {
                        this.responseFuture.completeExceptionally((Throwable)new RemoteExecutionException(response.getErrorMessage()));
                        this.span.log(response.getErrorMessage());
                        this.failedMeter.mark();
                    } else {
                        this.responseFuture.complete(response);
                    }
                    this.isProcessed = true;
                    this.responseSize.update(message.getBytes().length);
                } else {
                    this.responseFuture.completeExceptionally((Throwable)new RequestTimedOutException((Throwable)new TimeoutException()));
                    this.span.setTag("timeout", "true");
                    this.failedMeter.mark();
                    KafkaRpcClientFactory.this.rpcResponseMap.remove(this.rpcId);
                    KafkaRpcClientFactory.this.messageCache.remove(this.rpcId);
                    KafkaRpcClientFactory.this.currentChunkCache.remove(this.rpcId);
                }
                this.rpcDuration.update(System.currentTimeMillis() - this.requestCreationTime);
                this.span.finish();
            }
            catch (Throwable e) {
                LOG.warn("Error while handling response for RPC module: {}. Response string: {}", new Object[]{this.rpcModule.getId(), message, e});
            }
        }

        @Override
        public int compareTo(Delayed other) {
            long myDelay = this.getDelay(TimeUnit.MILLISECONDS);
            long otherDelay = other.getDelay(TimeUnit.MILLISECONDS);
            return Long.compare(myDelay, otherDelay);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long now = System.currentTimeMillis();
            return unit.convert(this.expirationTime - now, TimeUnit.MILLISECONDS);
        }

        @Override
        public boolean isProcessed() {
            return this.isProcessed;
        }

        @Override
        public String getRpcId() {
            return this.rpcId;
        }
    }
}

