/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.common.util.buffer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StreamCorruptedException;
import java.util.function.IntUnaryOperator;
import java.util.logging.Level;
import org.apache.sshd.common.PropertyResolver;
import org.apache.sshd.common.util.GenericUtils;
import org.apache.sshd.common.util.NumberUtils;
import org.apache.sshd.common.util.ValidateUtils;
import org.apache.sshd.common.util.buffer.Buffer;
import org.apache.sshd.common.util.io.IoUtils;
import org.apache.sshd.common.util.logging.SimplifiedLog;

public final class BufferUtils {
    public static final char DEFAULT_HEX_SEPARATOR = ' ';
    public static final char EMPTY_HEX_SEPARATOR = '\u0000';
    public static final String HEX_DIGITS = "0123456789abcdef";
    public static final String HEXDUMP_CHUNK_SIZE = "sshd-hexdump-chunk-size";
    public static final int DEFAULT_HEXDUMP_CHUNK_SIZE = 64;
    public static final Level DEFAULT_HEXDUMP_LEVEL = Level.FINEST;
    public static final IntUnaryOperator DEFAULT_BUFFER_GROWTH_FACTOR = BufferUtils::getNextPowerOf2;
    public static final long MAX_UINT32_VALUE = 0xFFFFFFFFL;
    public static final int MAX_UINT8_VALUE = 255;

    private BufferUtils() {
        throw new UnsupportedOperationException("No instance allowed");
    }

    public static void dumpHex(SimplifiedLog logger, Level level, String prefix, PropertyResolver resolver, char sep, byte ... data) {
        BufferUtils.dumpHex(logger, level, prefix, resolver, sep, data, 0, NumberUtils.length(data));
    }

    public static void dumpHex(SimplifiedLog logger, Level level, String prefix, PropertyResolver resolver, char sep, byte[] data, int offset, int len) {
        BufferUtils.dumpHex(logger, level, prefix, sep, resolver.getIntProperty(HEXDUMP_CHUNK_SIZE, 64), data, offset, len);
    }

    public static void dumpHex(SimplifiedLog logger, Level level, String prefix, char sep, int chunkSize, byte ... data) {
        BufferUtils.dumpHex(logger, level, prefix, sep, chunkSize, data, 0, NumberUtils.length(data));
    }

    public static void dumpHex(SimplifiedLog logger, Level level, String prefix, char sep, int chunkSize, byte[] data, int offset, int len) {
        if (logger == null || level == null || !logger.isEnabled(level)) {
            return;
        }
        StringBuilder sb = new StringBuilder(chunkSize * 3 + prefix.length() + 64);
        sb.append(prefix);
        int remainLen = len;
        int chunkIndex = 1;
        int curOffset = offset;
        int totalLen = 0;
        while (remainLen > 0) {
            sb.setLength(prefix.length());
            sb.append(" [chunk #").append(chunkIndex).append(']');
            int dumpSize = Math.min(chunkSize, remainLen);
            sb.append('(').append(totalLen += dumpSize).append('/').append(len).append(')');
            try {
                BufferUtils.appendHex(sb.append(' '), data, curOffset, dumpSize, sep);
            }
            catch (IOException e) {
                sb.append(e.getClass().getSimpleName()).append(": ").append(e.getMessage());
            }
            for (int index = dumpSize; index < chunkSize; ++index) {
                if (sep != '\u0000') {
                    sb.append(' ');
                }
                sb.append("  ");
            }
            sb.append("    ");
            int pos = curOffset;
            for (int l = 0; l < dumpSize; ++l) {
                int b = data[pos] & 0xFF;
                if (b > 32 && b < 126) {
                    sb.append((char)b);
                } else {
                    sb.append('.');
                }
                ++pos;
            }
            logger.log(level, sb.toString());
            remainLen -= dumpSize;
            curOffset += dumpSize;
            ++chunkIndex;
        }
    }

    public static String toHex(byte ... array) {
        return BufferUtils.toHex(array, 0, NumberUtils.length(array));
    }

    public static String toHex(char sep, byte ... array) {
        return BufferUtils.toHex(array, 0, NumberUtils.length(array), sep);
    }

    public static String toHex(byte[] array, int offset, int len) {
        return BufferUtils.toHex(array, offset, len, ' ');
    }

    public static String toHex(byte[] array, int offset, int len, char sep) {
        if (len <= 0) {
            return "";
        }
        try {
            return BufferUtils.appendHex(new StringBuilder(len * 3), array, offset, len, sep).toString();
        }
        catch (IOException e) {
            return e.getClass().getSimpleName() + ": " + e.getMessage();
        }
    }

    public static <A extends Appendable> A appendHex(A sb, char sep, byte ... array) throws IOException {
        return BufferUtils.appendHex(sb, array, 0, NumberUtils.length(array), sep);
    }

    public static <A extends Appendable> A appendHex(A sb, byte[] array, int offset, int len, char sep) throws IOException {
        if (len <= 0) {
            return sb;
        }
        int maxOffset = offset + len;
        for (int curOffset = offset; curOffset < maxOffset; ++curOffset) {
            byte b = array[curOffset];
            if (curOffset > offset && sep != '\u0000') {
                sb.append(sep);
            }
            sb.append(HEX_DIGITS.charAt(b >> 4 & 0xF));
            sb.append(HEX_DIGITS.charAt(b & 0xF));
        }
        return sb;
    }

    public static byte[] decodeHex(char separator, CharSequence csq) {
        return BufferUtils.decodeHex(separator, csq, 0, GenericUtils.length(csq));
    }

    public static byte[] decodeHex(char separator, CharSequence csq, int start, int end) {
        byte[] bytes;
        int len = end - start;
        ValidateUtils.checkTrue(len >= 0, "Bad HEX sequence length: %d", len);
        if (len == 0) {
            return GenericUtils.EMPTY_BYTE_ARRAY;
        }
        int delta = 2;
        if (separator != '\u0000') {
            ValidateUtils.checkTrue(len % 3 == 2, "Invalid separated HEX sequence length: %d", len);
            bytes = new byte[(len + 1) / 3];
            ++delta;
        } else {
            ValidateUtils.checkTrue((len & 1) == 0, "Invalid contiguous HEX sequence length: %d", len);
            bytes = new byte[len >>> 1];
        }
        int writeLen = 0;
        int curPos = start;
        while (curPos < end) {
            bytes[writeLen] = BufferUtils.fromHex(csq.charAt(curPos), csq.charAt(curPos + 1));
            curPos += delta;
            ++writeLen;
        }
        assert (writeLen == bytes.length);
        return bytes;
    }

    public static <S extends OutputStream> int decodeHex(S stream, char separator, CharSequence csq) throws IOException {
        return BufferUtils.decodeHex(stream, separator, csq, 0, GenericUtils.length(csq));
    }

    public static <S extends OutputStream> int decodeHex(S stream, char separator, CharSequence csq, int start, int end) throws IOException {
        int len = end - start;
        ValidateUtils.checkTrue(len >= 0, "Bad HEX sequence length: %d", len);
        int delta = 2;
        if (separator != '\u0000') {
            ValidateUtils.checkTrue(len % 3 == 2, "Invalid separated HEX sequence length: %d", len);
            ++delta;
        } else {
            ValidateUtils.checkTrue((len & 1) == 0, "Invalid contiguous HEX sequence length: %d", len);
        }
        int writeLen = 0;
        int curPos = start;
        while (curPos < end) {
            stream.write(BufferUtils.fromHex(csq.charAt(curPos), csq.charAt(curPos + 1)) & 0xFF);
            curPos += delta;
            ++writeLen;
        }
        return writeLen;
    }

    public static byte fromHex(char hi, char lo) throws NumberFormatException {
        int hiValue = HEX_DIGITS.indexOf(hi >= 65 && hi <= 70 ? 97 + (hi - 65) : hi);
        int loValue = HEX_DIGITS.indexOf(lo >= 65 && lo <= 70 ? 97 + (lo - 65) : lo);
        if (hiValue < 0 || loValue < 0) {
            throw new NumberFormatException("fromHex(" + new String(new char[]{hi, lo}) + ") non-HEX characters");
        }
        return (byte)((hiValue << 4) + loValue);
    }

    public static int readInt(InputStream input, byte[] buf) throws IOException {
        return BufferUtils.readInt(input, buf, 0, NumberUtils.length(buf));
    }

    public static int readInt(InputStream input, byte[] buf, int offset, int len) throws IOException {
        return (int)BufferUtils.readUInt(input, buf, offset, len);
    }

    public static long readUInt(InputStream input, byte[] buf) throws IOException {
        return BufferUtils.readUInt(input, buf, 0, NumberUtils.length(buf));
    }

    public static long readUInt(InputStream input, byte[] buf, int offset, int len) throws IOException {
        try {
            if (len < 4) {
                throw new IllegalArgumentException("Not enough data for a UINT: required=4, available=" + len);
            }
            IoUtils.readFully(input, buf, offset, 4);
            return BufferUtils.getUInt(buf, offset, len);
        }
        catch (Error | RuntimeException e) {
            throw new StreamCorruptedException("Failed (" + e.getClass().getSimpleName() + ")" + " to read UINT value: " + e.getMessage());
        }
    }

    public static long getUInt(byte ... buf) {
        return BufferUtils.getUInt(buf, 0, NumberUtils.length(buf));
    }

    public static long getUInt(byte[] buf, int off, int len) {
        if (len < 4) {
            throw new IllegalArgumentException("Not enough data for a UINT: required=4, available=" + len);
        }
        long l = (long)(buf[off] << 24) & 0xFF000000L;
        l |= (long)(buf[off + 1] << 16) & 0xFF0000L;
        l |= (long)(buf[off + 2] << 8) & 0xFF00L;
        return l |= (long)buf[off + 3] & 0xFFL;
    }

    public static void writeInt(OutputStream output, int value, byte[] buf) throws IOException {
        BufferUtils.writeUInt(output, value, buf, 0, NumberUtils.length(buf));
    }

    public static void writeInt(OutputStream output, int value, byte[] buf, int off, int len) throws IOException {
        BufferUtils.writeUInt(output, (long)value & 0xFFFFFFFFL, buf, off, len);
    }

    public static void writeUInt(OutputStream output, long value, byte[] buf) throws IOException {
        BufferUtils.writeUInt(output, value, buf, 0, NumberUtils.length(buf));
    }

    public static void writeUInt(OutputStream output, long value, byte[] buf, int off, int len) throws IOException {
        try {
            int writeLen = BufferUtils.putUInt(value, buf, off, len);
            output.write(buf, off, writeLen);
        }
        catch (Error | RuntimeException e) {
            throw new StreamCorruptedException("Failed (" + e.getClass().getSimpleName() + ")" + " to write UINT value=" + value + ": " + e.getMessage());
        }
    }

    public static int putUInt(long value, byte[] buf) {
        return BufferUtils.putUInt(value, buf, 0, NumberUtils.length(buf));
    }

    public static int putUInt(long value, byte[] buf, int off, int len) {
        if (len < 4) {
            throw new IllegalArgumentException("Not enough data for a UINT: required=4, available=" + len);
        }
        buf[off] = (byte)(value >> 24 & 0xFFL);
        buf[off + 1] = (byte)(value >> 16 & 0xFFL);
        buf[off + 2] = (byte)(value >> 8 & 0xFFL);
        buf[off + 3] = (byte)(value & 0xFFL);
        return 4;
    }

    public static boolean equals(byte[] a1, byte[] a2) {
        int len2;
        int len1 = NumberUtils.length(a1);
        if (len1 != (len2 = NumberUtils.length(a2))) {
            return false;
        }
        return BufferUtils.equals(a1, 0, a2, 0, len1);
    }

    public static boolean equals(byte[] a1, int a1Offset, byte[] a2, int a2Offset, int length) {
        int len1 = NumberUtils.length(a1);
        int len2 = NumberUtils.length(a2);
        if (len1 < a1Offset + length || len2 < a2Offset + length) {
            return false;
        }
        while (length-- > 0) {
            if (a1[a1Offset++] == a2[a2Offset++]) continue;
            return false;
        }
        return true;
    }

    public static int getNextPowerOf2(int value) {
        return value < 8 ? 8 : NumberUtils.getNextPowerOf2(value);
    }

    public static int updateLengthPlaceholder(Buffer buffer, int lenPos) {
        int startPos = lenPos + 4;
        int endPos = buffer.wpos();
        int dataLength = endPos - startPos;
        ValidateUtils.checkTrue(dataLength >= 0, "Illegal data length: %d", dataLength);
        buffer.wpos(lenPos);
        buffer.putInt(dataLength);
        buffer.wpos(endPos);
        return dataLength;
    }

    public static void updateLengthPlaceholder(Buffer buffer, int lenPos, int dataLength) {
        int curPos = buffer.wpos();
        buffer.wpos(lenPos);
        buffer.putInt(dataLength);
        buffer.wpos(curPos);
    }

    public static <B extends Buffer> B clear(B buffer) {
        if (buffer != null) {
            buffer.clear();
        }
        return buffer;
    }

    public static long validateInt32Value(long value, String message) {
        ValidateUtils.checkTrue(BufferUtils.isValidInt32Value(value), message, value);
        return value;
    }

    public static long validateInt32Value(long value, String format, Object arg) {
        ValidateUtils.checkTrue(BufferUtils.isValidInt32Value(value), format, arg);
        return value;
    }

    public static long validateInt32Value(long value, String format, Object ... args) {
        ValidateUtils.checkTrue(BufferUtils.isValidInt32Value(value), format, args);
        return value;
    }

    public static boolean isValidInt32Value(long value) {
        return value >= Integer.MIN_VALUE && value <= Integer.MAX_VALUE;
    }

    public static long validateUint32Value(long value, String message) {
        ValidateUtils.checkTrue(BufferUtils.isValidUint32Value(value), message, value);
        return value;
    }

    public static long validateUint32Value(long value, String format, Object arg) {
        ValidateUtils.checkTrue(BufferUtils.isValidUint32Value(value), format, arg);
        return value;
    }

    public static long validateUint32Value(long value, String format, Object ... args) {
        ValidateUtils.checkTrue(BufferUtils.isValidUint32Value(value), format, args);
        return value;
    }

    public static boolean isValidUint32Value(long value) {
        return value >= 0L && value <= 0xFFFFFFFFL;
    }
}

