/* Xmp.java

   COPYRIGHT 2004 KRUPCZAK.ORG, LLC.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA
 
   For more information, visit:
   http://www.krupczak.org/
*/

package org.krupczak.Xmp;

/** 
 *  Xmp is the base class for the <b>X</b>ML <b>M</b>anagement <b>P</b>rotocol;
 *  it contains definitions, message types, error codes, and syntax
 *  definitions.  It also contains class methods for converting between 
 *  various protocol values and their ASCII string equivalents
 *  @author Bobby Krupczak, rdk@krupczak.org
 *  @version $Id: Xmp.java 15 2008-07-17 14:20:37Z rdk $
 *  @see XmpSession
 *  @see XmpMessage
 **/

public final class Xmp {

    /** IANA assigned UDP/TCP port used for agents
     *  managers do not have a reserved, well-known port
     *  due to our de-emphasis on managers in the Cartographer/XMP
     *  architecture 
    */
    public static final int XMP_PORT = 5270;

    /** current version **/
    public static final String VERSION = "Xmp (v1.0)";
    public static final int XMP_VERSION = 1;

    /** XMP Message types (internal representation) **/
    public static final int MSGTYPE_INVALID = 0;
    public static final int MSGTYPE_RESPONSE = 1;
    public static final int MSGTYPE_GETREQUEST = 2;
    public static final int MSGTYPE_SETREQUEST = 3;
    public static final int MSGTYPE_SELECTTABLEREQUEST = 4;
    public static final int MSGTYPE_INSERTTABLEREQUEST = 5;
    public static final int MSGTYPE_DELETETABLEREQUEST = 6;
    public static final int MSGTYPE_UPDATETABLEREQUEST = 7;
    public static final int MSGTYPE_TRAP = 8;
    public static final int MSGTYPE_INFORMATION = 9;

    /** XMP Message type as conveyed in XML PDU **/
    public static final String messageTypeToString(int msgType) {
       switch (msgType) {

         case MSGTYPE_RESPONSE:
            return new String("Response");
         case MSGTYPE_GETREQUEST:
            return new String("GetRequest");
         case MSGTYPE_SETREQUEST:
            return new String("SetRequest");
         case MSGTYPE_SELECTTABLEREQUEST:
            return new String("SelectTableRequest");
         case MSGTYPE_INSERTTABLEREQUEST:
            return new String("InsertTableRequest");
         case MSGTYPE_DELETETABLEREQUEST:
            return new String("DeleteTableRequest");
         case MSGTYPE_UPDATETABLEREQUEST:
            return new String("UpdateTableRequest");
         case MSGTYPE_TRAP: 
            return new String("Trap");
         case MSGTYPE_INFORMATION: 
            return new String("Information");
         default:
            return null;
       }
    }
    public static final int stringToMessageType(String msgType) {
        if (msgType == null)
	   return MSGTYPE_INVALID;
	if (msgType.equals("Response"))
	   return MSGTYPE_RESPONSE;
	if (msgType.equals("GetRequest"))
	   return MSGTYPE_RESPONSE;
	if (msgType.equals("SetRequest"))
	   return MSGTYPE_SETREQUEST;
	if (msgType.equals("SelectTableRequest"))
	   return MSGTYPE_SELECTTABLEREQUEST;
	if (msgType.equals("InsertTableRequest"))
	   return MSGTYPE_INSERTTABLEREQUEST;
	if (msgType.equals("DeleteTableRequest"))
	   return MSGTYPE_DELETETABLEREQUEST;
	if (msgType.equals("UpdateTableRequest"))
	   return MSGTYPE_UPDATETABLEREQUEST;
	if (msgType.equals("Trap"))
	   return MSGTYPE_TRAP;
	if (msgType.equals("Information"))
	   return MSGTYPE_INFORMATION;
        return MSGTYPE_INVALID;
    }

    public static boolean isTableOperation(String msgType) { 
       if (msgType.equals("SelectTableRequest"))
	  return true;
       if (msgType.equals("InsertTableRequest"))
	  return true;
       if (msgType.equals("DeleteTableRequest"))
	  return true;
       if (msgType.equals("UpdateTableRequest"))
	  return true;
        return false;
    }

    public static boolean isTableOperation(int msgType) {

	if ((msgType >= MSGTYPE_SELECTTABLEREQUEST) &&
            (msgType <= MSGTYPE_UPDATETABLEREQUEST))
           return true;
        return false;
    }

    /** XMP error status values, internal representation **/
    public static final int ERROR_INVALID = -1;
    public static final int ERROR_NOERROR = 0;
    public static final int ERROR_MOREDATA = 1;
    public static final int ERROR_TOOBIG = 2;
    public static final int ERROR_NOSUCHNAME = 3;
    public static final int ERROR_BADVALUE = 4;
    public static final int ERROR_READONLY = 5;
    public static final int ERROR_GENERROR = 6;
    public static final int ERROR_NOACCESS = 7;
    public static final int ERROR_NOTTABULAR = 8;
    public static final int ERROR_WRONGTYPE = 9;
    public static final int ERROR_WRONGLENGTH = 10;
    public static final int ERROR_WRONGENCODING = 11;
    public static final int ERROR_WRONGVALUE = 12;
    public static final int ERROR_NOCREATION = 13;
    public static final int ERROR_INCONSISTENTVALUE = 14;
    public static final int ERROR_RESOURCEUNAVAILABLE = 15;
    public static final int ERROR_COMMITFAILED = 16;
    public static final int ERROR_UNDOFAILED = 17;
    public static final int ERROR_AUTHORIZATIONERROR = 18;
    public static final int ERROR_NOTWRITABLE = 19;
    public static final int ERROR_INCONSISTENTNAME = 20;
    public static final int ERROR_PARSEERROR = 21;

    /** XMP error status as conveyed in XML PDU **/
    public static final String errorStatusToString(int errorStatus) {
       switch (errorStatus) {
         case ERROR_NOERROR:
            return new String("noError");
         case ERROR_MOREDATA:
            return new String("moreData");
         case ERROR_TOOBIG:
            return new String("tooBig");
         case ERROR_NOSUCHNAME:
            return new String("noSuchName");
         case ERROR_BADVALUE:
            return new String("badValue");
         case ERROR_READONLY:
            return new String("readOnly");
         case ERROR_GENERROR:
            return new String("genError");
         case ERROR_NOACCESS:
            return new String("noAccess");
         case ERROR_NOTTABULAR:
            return new String("notTabular");
         case ERROR_WRONGTYPE:
            return new String("wrongType");
         case ERROR_WRONGLENGTH:
            return new String("wrongLength");
         case ERROR_WRONGENCODING:
            return new String("wrongEncoding");
         case ERROR_WRONGVALUE:
            return new String("wrongValue");
         case ERROR_NOCREATION:
            return new String("noCreation");
         case ERROR_INCONSISTENTVALUE:
            return new String("inconsistentValue");
         case ERROR_RESOURCEUNAVAILABLE:
            return new String("resourceUnavailable");
         case ERROR_COMMITFAILED:
            return new String("commitFailed");
         case ERROR_UNDOFAILED:
            return new String("undoFailed");
         case ERROR_AUTHORIZATIONERROR:
            return new String("authorizationError");
         case ERROR_NOTWRITABLE:
            return new String("notWritable");
         case ERROR_INCONSISTENTNAME:
            return new String("inconsistentName");
         case ERROR_PARSEERROR:
            return new String("parseError");
         default:
	    return null;
       }
    }

    public static final int stringToErrorStatus(String errorStatus) {
       if (errorStatus == null)
          return ERROR_INVALID;

       if (errorStatus.equals("noError"))
	  return ERROR_NOERROR;
       if (errorStatus.equals("moreData"))
	  return ERROR_MOREDATA;
       if (errorStatus.equals("tooBig"))
	  return ERROR_TOOBIG;
       if (errorStatus.equals("noSuchName"))
	  return ERROR_NOSUCHNAME;
       if (errorStatus.equals("badValue"))
	  return ERROR_BADVALUE;
       if (errorStatus.equals("readOnly"))
	  return ERROR_READONLY;
       if (errorStatus.equals("genError"))
	  return ERROR_GENERROR;
       if (errorStatus.equals("noAccess"))
	  return ERROR_NOACCESS;
       if (errorStatus.equals("notTabular"))
	  return ERROR_NOTTABULAR;
       if (errorStatus.equals("wrongType"))
	  return ERROR_WRONGTYPE;
       if (errorStatus.equals("wrongLength"))
	  return ERROR_WRONGLENGTH;
       if (errorStatus.equals("wrongEncoding"))
	  return ERROR_WRONGENCODING;
       if (errorStatus.equals("wrongValue"))
	  return ERROR_WRONGVALUE;
       if (errorStatus.equals("noCreation"))
	  return ERROR_NOCREATION;
       if (errorStatus.equals("inconsistentValue"))
	  return ERROR_INCONSISTENTVALUE;
       if (errorStatus.equals("resourceUnavailable"))
	  return ERROR_RESOURCEUNAVAILABLE;
       if (errorStatus.equals("commitFailed"))
	  return ERROR_COMMITFAILED;
       if (errorStatus.equals("undoFailed"))
	  return ERROR_UNDOFAILED;
       if (errorStatus.equals("authorizationError"))
	  return ERROR_AUTHORIZATIONERROR;
       if (errorStatus.equals("notWritable"))
	  return ERROR_NOTWRITABLE;
       if (errorStatus.equals("inconsistentName"))
	  return ERROR_INCONSISTENTNAME;

       return ERROR_INVALID;
    }

    /** XMP syntax values (internal representation) **/
    public static final int SYNTAX_INVALID = 0;
    public static final int SYNTAX_COUNTER = 1;
    public static final int SYNTAX_IPV4ADDRESS = 2;
    public static final int SYNTAX_IPV6ADDRESS = 3;
    public static final int SYNTAX_DATETIME = 4;
    public static final int SYNTAX_GAUGE = 5;
    public static final int SYNTAX_INTEGER = 6;
    public static final int SYNTAX_UNSIGNEDINTEGER = 7;
    public static final int SYNTAX_BOOLEAN = 8;
    public static final int SYNTAX_MACADDRESS = 9;
    public static final int SYNTAX_PHYSADDRESS = 10;
    public static final int SYNTAX_DISPLAYSTRING = 11;
    public static final int SYNTAX_BINARYSTRING = 12;
    public static final int SYNTAX_NULLSYNTAX = 13;
    public static final int SYNTAX_TABLE = 14;
    public static final int SYNTAX_FLOATINGPOINT = 15;
    public static final int SYNTAX_UNSUPPORTEDVAR = 16;
    public static final int SYNTAX_EXTENDEDBOOLEAN = 17;

    /** XMP syntax values as conveyed in XML PDU **/
    public static final String syntaxToString(int syntax) {
       switch (syntax) {
         case SYNTAX_COUNTER:
	    return new String("counter");
         case SYNTAX_IPV4ADDRESS:
	    return new String("ipV4Address");
         case SYNTAX_IPV6ADDRESS:
	    return new String("ipV6Address");
         case SYNTAX_DATETIME:
	    return new String("dateTime");
         case SYNTAX_GAUGE:
	    return new String("gauge");
         case SYNTAX_INTEGER:
	    return new String("integer");
         case SYNTAX_UNSIGNEDINTEGER:
	    return new String("unsignedInteger");
         case SYNTAX_BOOLEAN:
	    return new String("bool");
         case SYNTAX_MACADDRESS:
	    return new String("macAddress");
         case SYNTAX_PHYSADDRESS:
	    return new String("physAddress");
         case SYNTAX_DISPLAYSTRING:
	    return new String("displayString");
         case SYNTAX_BINARYSTRING:
	    return new String("binaryString");
         case SYNTAX_NULLSYNTAX:
	    return new String("nullSyntax");
         case SYNTAX_TABLE:
	    return new String("table");
         case SYNTAX_FLOATINGPOINT:
	    return new String("floatingPoint");
         case SYNTAX_UNSUPPORTEDVAR:
	    return new String("unsupportedVariable");
         case SYNTAX_EXTENDEDBOOLEAN:
	    return new String("extendedBoolean");
         default:
            return null;

       }
    }
 
    public static final int stringToSyntax(String syntax) {
	if (syntax == null)
	   return SYNTAX_INVALID;

        if (syntax.equals("counter"))
	   return SYNTAX_COUNTER;
        if (syntax.equals("ipV4Address"))
	   return SYNTAX_IPV4ADDRESS;
        if (syntax.equals("ipV6Address"))
	   return SYNTAX_IPV6ADDRESS;
        if (syntax.equals("dateTime"))
	   return SYNTAX_DATETIME;
        if (syntax.equals("gauge"))
	   return SYNTAX_GAUGE;
        if (syntax.equals("integer"))
	   return SYNTAX_INTEGER;
        if (syntax.equals("unsignedInteger"))
	   return SYNTAX_UNSIGNEDINTEGER;
        if (syntax.equals("bool"))
	   return SYNTAX_BOOLEAN;
        if (syntax.equals("macAddress"))
	   return SYNTAX_MACADDRESS;
        if (syntax.equals("physAddress"))
	   return SYNTAX_PHYSADDRESS;
        if (syntax.equals("displayString"))
	   return SYNTAX_DISPLAYSTRING;
        if (syntax.equals("binaryString"))
	   return SYNTAX_BINARYSTRING;
        if (syntax.equals("nullSyntax"))
	   return SYNTAX_NULLSYNTAX;
        if (syntax.equals("table"))
	   return SYNTAX_TABLE;
        if (syntax.equals("floatingPoint"))
	   return SYNTAX_FLOATINGPOINT;
        if (syntax.equals("unsupportedVariable"))
	   return SYNTAX_UNSUPPORTEDVAR;
        if (syntax.equals("extendedBoolean"))
	   return SYNTAX_EXTENDEDBOOLEAN;

        return SYNTAX_INVALID;
    }

} /* class Xmp */
